### A Pluto.jl notebook ###
# v0.20.20

using Markdown
using InteractiveUtils

# This Pluto notebook uses @bind for interactivity. When running this notebook outside of Pluto, the following 'mock version' of @bind gives bound variables a default value (instead of an error).
macro bind(def, element)
    #! format: off
    return quote
        local iv = try Base.loaded_modules[Base.PkgId(Base.UUID("6e696c72-6542-2067-7265-42206c756150"), "AbstractPlutoDingetjes")].Bonds.initial_value catch; b -> missing; end
        local el = $(esc(element))
        global $(esc(def)) = Core.applicable(Base.get, el) ? Base.get(el) : iv(el)
        el
    end
    #! format: on
end

# ╔═╡ a145e2fe-a903-11eb-160f-df7ea83fa3e6
begin
	using HypertextLiteral, PlutoUI, CSV , ColorSchemes 
	using DataFrames, Dates, PeriodicalDates, StatsBase, PlutoPlotly
end

# ╔═╡ 1c9ca8cc-71d4-452e-a181-2574e8a1ac4d
html"""
<picture>
  <source srcset="https://gci.iscte-iul.pt/2022/identidade/files/brand_v2/escolas/rgb/ibs/en/horizontal/negative/rgb_ibs_en_horizontal_negative.png" media="(prefers-color-scheme: dark)" width="240px!important" style="float:right;">
  <img src="https://gci.iscte-iul.pt/2022/identidade/files/brand_v2/escolas/rgb/ibs/en/horizontal/main/rgb_ibs_en_horizontal_main.png" width="240px" style="float:right;">
</picture>
"""

# ╔═╡ 943edb36-acc8-4b2a-94ab-c544f9eb279b
md"""
# Week 11 - Fiscal Policy and the Government Budget

# Exercises

**Macroeconomics, ISCTE-IUL**
"""

# ╔═╡ 1245aef9-18ce-42f5-af82-c2e6cce70798
md"""
**Vivaldo Mendes, Ricardo Gouveia-Mendes, Luís Clemente-Casinhas**

**December 2025**
"""

# ╔═╡ a66961e9-d653-414d-b81f-ed924c07007b
md"""
---
"""

# ╔═╡ a8b69b08-eaee-4b1d-baed-394069f62188
begin
	function Base.show(io::IO, mimetype::MIME"text/html", plt::PlotlyBase.Plot)
       # Remove responsive flag on the plot as we handle responsibity via ResizeObeserver and leaving it on makes the div flickr during updates
	hasproperty(plt,:config) && plt.config.responsive && (plt.config.responsive = false)   
	show(io,mimetype, @htl("""
			<div>
			<script id=asdf>
			const {plotly} = await import("https://cdn.plot.ly/plotly-2.2.0.min.js")
			const PLOT = this ?? document.createElement("div");
		

		
			Plotly.react(PLOT, $(HypertextLiteral.JavaScript(PlotlyBase.json(plt))));


		
			const pluto_output = currentScript.parentElement.closest('pluto-output')

			const resizeObserver = new ResizeObserver(entries => {
				Plotly.Plots.resize(PLOT)
			})

			resizeObserver.observe(pluto_output)

			invalidation.then(() => {
				resizeObserver.disconnect()
			})
		
			return PLOT
			</script>
			</div>
	"""))
	end
end

# ╔═╡ 4618a663-f77e-4ccc-9b00-e84dbd5beb34
md"""
### Packages used in this notebook
"""

# ╔═╡ 4bc7654b-1019-48bc-8524-4ee77e1ed577
md"""
---
"""

# ╔═╡ dcd13560-0f73-4aaa-b59a-b7961d8cdc63
md"""
## Exercise 1. Primary deficit
"""

# ╔═╡ df9417d2-ebe1-498c-b299-f2b8fdee1084
md"""
*From the textbook*

Suppose government purchases of Goods&Services amount to \$2.5 trillion, transfer payments to households amount to \$1 trillion, net interest payments are \$0.5 trillion, and tax revenue is valued at \$3 trillion.

👉 a) Calculate the government budget balance.

"""

# ╔═╡ d3349840-4c97-443c-9a50-f62ba4da3f17
md"""
!!! note

	The next cell passes the data of this exercise into the notebook. Use the empty cells below to perform the computations.

"""

# ╔═╡ ff6b8641-0e63-4f3e-9400-7979a1536e80
Ḡ = 2.5 ; T̄R̄ = 1.0 ; Interest = 0.5 ; T̄ = 3.0 ;

# ╔═╡ 42e4c961-1644-495d-96a0-38a12182a1f7
md"""
Obtain the government budget balance in the next cell:
"""

# ╔═╡ 7157f8e7-9771-4b13-846b-9a3f496e2455
Budget = T̄ - (Ḡ + T̄R̄ + Interest)

# ╔═╡ 26d25950-b6bf-44c8-9ee6-21102094d4c8
md"""
Obtain the government primary budget in the next cell:
"""

# ╔═╡ ea0ae94d-bd6a-4bf5-98b2-0a9dd28b1621
PrimaryBudget = Budget + Interest

# ╔═╡ bb024642-ce35-4464-83f8-af4493d8619b
md"""
!!! answer "Answer (a)"

	The government budget balance is equal to $-1.0$ trillion dollars.

"""

# ╔═╡ 5686fa97-4db8-4e41-9bf2-1ca265d88c91
md"""
👉 b) Calculate the government primary budget balance.
"""

# ╔═╡ d4af7521-c311-497a-ada1-3b0a74b5e411
md"""
!!! answer "Answer (b)"

	

	The government primary budget balance is equal to $-0.5$ trillion dollars. Notice that the primary BUDGET is negative. Usually, in this case we say that the primary DEFICIT is positive. That is, the primary deficit is equal to + 0.5 trillion.

"""

# ╔═╡ 5cef5110-edd0-4c7f-94a3-0d4b3d183c14
md"""
---
"""

# ╔═╡ 912da456-9d55-4986-8e6f-56f0d1df8d63
md"""
## Exercise 2. Population aging and the government deficit
"""

# ╔═╡ 0311c862-313e-4ab5-b94f-9119b34ef532
md"""
*Adapted from the textbook*

Assume that Social Security tax rates remain constant, but the number of employed people in the United States declines over time. Assume also that the proportion of the old population and their life expectancy increase.

👉 a) Explain the effect of such a scenario on the size of the contributions for social insurance and the government deficit in the United States.

"""

# ╔═╡ 2e73dd9a-d9a5-421a-8360-475a23259695
md"""
!!! tip "Answer (a)"

	The size of social contributions will decline over time, while the social security benefits will increase over time. The net result will be an increase in the Federal budget deficit due to the need to fill in the deficit of the social security balance.

"""

# ╔═╡ 08a12673-d5b4-4212-ad2d-9e32305bf335
md"""
👉 b) Assume now that employment remains constant but there is an increase in unemployment insurance benefits. How would your answer to part (a) change?

"""

# ╔═╡ 53f53bb1-9373-4efa-827b-8a5ea111697b
md"""
!!! tip "Answer (b)"

	The situation will get worse than in the previous question.

"""

# ╔═╡ d067df9c-476d-49c1-87f9-ed3037b32a7b
md"""
👉 c) What forces have been driving the evolution of total public spending in the US economy since the 1960s?
"""

# ╔═╡ b0b4285e-9051-4c20-a8f8-5524dc9ea3f4
md"""
!!! tip "Answer (c)"

	The aging of the population seems to be the factor that has driven the increase in public spending since the 1960s. In the following exercise, we will see if this aging factor is the real culprit of the alledged unsustainability of the social security system in the US.

"""

# ╔═╡ a4c3f332-2fb9-4230-8634-bfdec7a27aab
md"""
---
"""

# ╔═╡ 4cac2f5c-943d-423f-9f8e-e1a741ed1762
md"""
## Exercise 3. Aging in the USA and Portugal
"""

# ╔═╡ a9885086-22ac-4ed7-99da-40cfb107a8c8
md"""
The view that the population in Western countries is getting older and older as time passes has become very popular and largely dominant. We see it everywhere: in the news media, politics, and mainstream economic parlance. In the figure below, we present the population pyramids for the US and Portugal in the year 2024. These pyramids can be obtained [here.](https://www.census.gov/data-tools/demo/idb/#/dashboard?COUNTRY_YEAR=2024&COUNTRY_YR_ANIM=2060&CCODE_SINGLE=US&CCODE=US=)

👉 a) What information can we get from a population pyramid? Do you see any difference between Portugal and the USA?  

"""

# ╔═╡ 7fca52e8-3153-4e58-8436-0ea68d010587
Resource("https://modernmacro.org/resources/Vivaldo/figures/PopPyramids.png", :width => 1090)

# ╔═╡ 7a38bfd9-7d09-491c-9193-14655e8b2013
md"""
!!! tip "Answer (a)"

	A pyramid population gives an excellent perspective of a country's population structure over several decades in the future.

	The pyramid of Portugal shows that the total population will decline over time, and the proportion of the older population will increase.

	In the case of the USA, we get the opposite results.

"""

# ╔═╡ 52b48932-96f4-4da0-ba3c-b1abeee65285
md"""
👉 b) Which country do you expect to have a large proportion of the old population in the future?  
"""

# ╔═╡ 8759d076-cc13-4d22-b0e1-98242ae64e16
md"""
!!! tip "Answer (b)"

	Portugal.

"""

# ╔═╡ d58319c7-0156-4804-859c-99b2f8ba4d3b
md"""
👉 c) In the following plot, we present the expected evolution of the total population of the USA and Portugal until 1960. Given what we learned about the impact of aging on the sustainability of public debt, which country is more exposed to such problems?  
"""

# ╔═╡ 0848497e-60ea-4953-8dc3-5815d49df237
md"""
!!! tip "Answer (c)"

	Portugal is exposed to such problems because its total population is expected to decline, and the proportion of its older population will increase. The USA does not display any difficulties regarding those issues.

"""

# ╔═╡ 24105ebf-251d-4b6c-bcff-1f035559c2cf
md"""
---
"""

# ╔═╡ 75df3711-3b97-4d9a-9f61-38ef759a0983
md"""
## Exercise 4. The Size of public debt
"""

# ╔═╡ 0b024eaa-f6d1-4d5e-b110-f74f29237ed0
md"""
The evolution of public debt in five G7 economies (plus Portugal, Spain, and Belgium) is presented in the figure below.

👉 What happened to public debt when the big financial crisis struck in 2007? And when the Covid19 pandemic broke out?

"""

# ╔═╡ bd2a646a-7719-4fec-872a-85ef69039040
md"""
!!! tip "Answer (a)"

	When the big financial crisis struck in 2007 and the Covid19 pandemic broke out in 2020, the level of public debt in all these countries increased significantly.

"""

# ╔═╡ 3b1585a8-5daa-42ea-a26e-eb46993a0d53
md"""
👉 Why did the level of public debt increase so significantly in the face of those two terrible shocks (financial crisis and Covid19)?

"""

# ╔═╡ 2da29462-367e-4d38-a2b9-01ca32e2bb73
md"""
!!! tip "Answer (b)"

	The level of public debt increased in all countries because governments raised spending a lot to prevent a massive recession from happening in the case of the financial crisis or a humanity crisis of dramatic proportions in the case of COVID-19.

"""

# ╔═╡ 046ae576-7815-4bd0-970b-c993a1abe665
md"""
👉 Now that the two shocks seem to be overcome, what strikes you in terms of the behavior of public debt in this set of countries?
"""

# ╔═╡ 40a2d136-3ccc-4e56-9408-50ea8836f501
md"""
!!! tip "Answer (c)"

	At least, two things strike us after the two shocks seem to be over: (i) the levels of public debt are going down fast and very close together; (ii) the case of Germany, which has a much lower level of public debt. 

"""

# ╔═╡ 2f6946dc-4346-4c9c-bbe1-fe1761eb0c1b
md"""
---
"""

# ╔═╡ 2371b919-ab42-4dc2-b72e-7c5c8b3c648f
md"""
## Exercise 5. Mandatory vs discretionary spending
"""

# ╔═╡ 08b17da3-50a2-41ca-b75c-209b1c7489d8
md"""

The US Congressional Budget Office (CBO) published last June **_["The 2023 Long-Term Budget Outlook"](https://www.cbo.gov/publication/59331#category.name)_**. In this document, the CBO portrays a dark future for the sustainability of the US budget, with ever-larger budget deficits resulting from Medicare and Social Security spending, as shown in the figure below. To understand this problem correctly, the distinction between mandatory and discretionary spending is crucial.

"""

# ╔═╡ 270882ce-0740-4605-b92d-502edaf68f99
Resource("https://vivaldomendes.org/images/depot/Aging_problem.png", :width => 750)

# ╔═╡ dd58f4ec-59be-4488-8392-e5a9a7a81b03
md"""
👉 a) What is the difference between mandatory and discretionary public spending?

"""

# ╔═╡ 45c669d6-7e9d-4cdc-8f8a-8e4e9b4d6382
md"""
!!! tip "Answer (a)"

	Mandatory spending includes all expenditures associated with the social insurance functions of a modern state: social security (taking care of older population), basic health care, and temporary unemployment benefits. They are tough to cut off because they represent the basic pillars of advanced modern societies.

	They also include spending on the classical functions of the state, such as justice, order, and defense (although, if one spends too much on defense, some part of it may be discretionary).

	Discretionary spending involves all expenditures that can be reduced without a significant change in a modern state's social and political structures. Examples include military spending, spending on goods and services, and public investment.

"""

# ╔═╡ 9b452fdc-e5c0-4aaa-827d-9f4b8f59c794
md"""
👉 b) In the following figure, we plot the evolution of mandatory, discretionary, and the sum of the two as a percentage of GDP. Do you agree with the dim view of the CBO? Specifically, look at what happened between 1975 and 2007.
"""

# ╔═╡ 23de519c-06a4-4879-94a6-75816e191c0c
md"""
!!! tip "Suggested answer (b)"

	1. Mandatory public spending remained relatively stable between 1975 (9.4% of GDP) and 2007 (10.1% of GDP). This means that over 32 years, this type of public spending increased only 0.7% of GDP. This type of spending increased significantly after 2008, but in this case, it had nothing to do with the aging of the population. It was caused by the impact of a massive financial crisis in 2008 and the Covid19 pandemic in 2020. So, maybe the CBO is overshooting the costs associated with the aging process.

	2. Discretionary spending as a percentage of GDP has dramatically decreased since 1968, so apparently, there is plenty of room for maneuvering to reduce this type of public spending.

	3. Adding the two types of public spending leads to an interesting fact: as a percentage of GDP, they represented 17.1% in 1962, compared with 17.4% in 2007. For 45 years, mandatory and discretionary spending sums remained almost at the same level.

	4. Maybe the picture presented by CBO is missing something. The population's aging did not start in 2008.

"""

# ╔═╡ 3ada049d-a39e-4ef5-974c-3f114852cda3
md"""
---
"""

# ╔═╡ 35e486b8-3ef0-4c6f-b272-566319b63f10
md"""
## Exercise 6. Projections, Projections: What to do with them?
"""

# ╔═╡ d43754b1-e498-4e83-a96e-40b36f39d802
md"""

In 2011, President Barack Obama said: 

>_"If you look at the numbers, then Medicare, in particular, will run out of money, and we will not be able to sustain that program no matter how much taxes go up [...] We have an obligation to make sure that we make those changes that are required to make it sustainable over the long term."_ [here](https://obamawhitehouse.archives.gov/the-press-office/2011/07/11/press-conference-president)

A recent set of studies in the US has unveiled a remarkable novelty: the projections made in the late 2000s about Medicare in the US proved dramatically wrong. See an exposition [here](https://www.nytimes.com/interactive/2023/09/05/upshot/medicare-budget-threat-receded.html), from where we borrow the image below. In 2010, the CBO projected that Medicare spending per beneficiary would be \$22,006 in 2023, but instead, it is only \$12,459. The figure below shows the gap, representing an error corresponding to \$3.9 trillion (around 14.1% of GDP in 2023).

Given this evidence, what do you make of the CBO projections (for 30 years) presented in Exercise 4?

"""

# ╔═╡ 2c7add2a-0a48-4f03-85d7-d6af603405df
Resource("https://vivaldomendes.org/images/depot/Surprise_Projections.png", :width => 500)

# ╔═╡ 61b2eb9d-9050-4fcb-8a62-9fa9847cd311
md"""
!!! tip "Possible answers"

	1. Those projections were made with all the available information in 2010; they got it wrong because relevant information only arrived later. There was nothing the CBO could do otherwise.

	2. Those projections were not made with all the available information in 2010; they got it wrong because they overlooked relevant information already available. The CBO should have done better.

	3. If the CBO got its projection so wrong for a period of 10 years, we should be very cautious to follow a CBO's projection for a much more extended period (30 years). Do not make the same mistake twice.

	4. ...

"""

# ╔═╡ 18103d5d-b4d2-4c27-9a0c-909bc5bb3b08
md"""
---
"""

# ╔═╡ 359bd7ea-3494-41a1-b2d8-8c6a25d7a30e
md"""
## Exercise 7. The data _vs_ the popular view
"""

# ╔═╡ bac4b203-a7bc-47f9-93d3-daedbac11ef6
md"""

The popular view on the population aging problem and the sustainability of the social security system (and public debt, for the matter) seems to be something like this:

>_"The US social security system will blow up because the evidence has shown that, as time goes on, the number of contributors gets smaller and smaller, while the number of receivers gets bigger and bigger."_

In the figure below, we present evidence for the US economy (the data can be found [here](https://www.ssa.gov/oact/tr/2022/lr4b3.html#alt_h)), regarding the ratio of the number of contributors (workers) per beneficiary of the social security system. What is your opinion about the sustainability of the US social security system, **_as far as this ratio is concerned_**?

"""

# ╔═╡ eab8e772-694b-490b-a791-a4f77fa4e53d
md"""
!!! tip "Answer"

	In the US, the usual argument about the aging of the population seems to be more of a myth than an actual problem. Since the 1980s, the ratio of contributors to beneficiaries has stabilized in the US social security system, precisely the opposite of the message the popular doomsday view usually portrays. It will not be due to the aging population **_de per se_** that the system will get into financial troubles in the US. 

	The common mistake is that people look at the 1950s or 1960s and compare the situation then with the case in the 2000s. Then they make a linear extrapolation into the future. When we analyze something seriously, this procedure is a dangerous (but common) mistake — avoid it!

"""

# ╔═╡ 49d3bea9-96d3-4bc7-8fc9-e413f615055f
md"""
---
"""

# ╔═╡ f805a155-e72c-458d-b7fc-dcc17a8694f3
md"""
## Exercise 8. Business cycles and primary balances
"""

# ╔═╡ 6faef3d2-5e28-4c9b-8238-d8882eabe6cf
md"""

👉 a) By inspecting Figure 4 below, what can one conclude about the primary balances of these sixteen countries of the OECD? **_Hint: we suggest to zoom in between 2004 and 2022._**

"""

# ╔═╡ f535efe2-5f46-4ab2-a3b8-a6968f558bbb
md"""
!!! tip "Answer (a)"

	The primary balances in all countries evolve with the same pattern. Firstly, all primary deficits improved up to 2008, then all went down abruptly to face the negative impact of the Great Recession of 2008-2009, then improved again in a gradual manner up to 2019. Finally, Covid19 hit all economies, and we know what has happened very well. 


"""

# ╔═╡ 46580728-1317-477d-baa8-8672aee50694
md"""

👉 b) What kind of information can we obtain in Figure 5 below regarding the two periods considered in the plot? 


"""

# ╔═╡ 06564a25-6d37-4861-9e86-0e37b8a36606
md"""
!!! tip "Answer (b)"

	There are two essential features in the second plot. As expected, the average primary balance is more negative for the entire period (1978-2022) than the average between 1978-2008. The reason is simple: fighting the Great Recession and the Covid19 pandemic involved much public spending. However, this fact does not happen in three countries: Finland. Denmark and Belgium (countries that ran average surpluses in 1978-2008).

"""

# ╔═╡ 023aa779-cc6d-48d3-8ced-a1f24e0aef35
md"""
👉 c) South Korea has the highest primary deficit in this sample of countries (an average of $-3.99\%$ of GDP). With such a high primary deficit average, what would we expect to see in terms of this country's total public debt as a percentage of GDP?


"""

# ╔═╡ 015f2c16-2320-426f-83bb-841c1d29213c
md"""
!!! tip "Answer (c)"

	By mere intuition, if a country runs an average primary deficit close 3.4% of GDP for 45 years, we would be inclined to think that the level of public debt as a percentage of GDP is expected to be very high.

"""

# ╔═╡ 86cc6fca-df27-464a-92ae-65edca20f1ff
md"""
👉 d) However, when looking at the total public debt of South Korea (59% of GDP in 2020), we find that it is one of the lowest levels in the whole OECD. What factor can explain this apparent contradiction between high primary deficits and low public debt?
"""

# ╔═╡ b8ef96d5-ec35-4d0c-aef7-bef214858661
md"""
!!! tip "Answer (d)"

	With such high negative values of the primary deficit as a percentage of GDP and having one of the lowest levels of Public debt as a percentage of GDP in the entire OECD, there must be some other macroeconomic variables that explain this apparent contradiction. As we will see in the next exercise, the force that explains that apparent contradiction is the growth rate of real GDP.

"""

# ╔═╡ bbd4a839-223a-4d74-a7ea-309a6036e65d
md"""
---
"""

# ╔═╡ 162d4e53-51a7-45be-a238-e84094960d4c
md"""
## Exercise 9. The sustainability of public debt
"""

# ╔═╡ 45a1b8d3-cf21-4b2c-abc6-b8530ce44947
md"""
Consider an economy with the following averages for the primary deficit as a percentage of GDP $(p)$, the real interest rate on public debt $(r_p)$, the rate of growth of real GDP $(g)$, and an initial Public Debt to GDP ratio $(d_1)$:

$$p=2 \ , \ r_p=1.2 \ , \ g=3 \ , \ d_1 =0 \qquad \text{(all in percentage points)}$$

These values represent a close picture of what happened in the US economy from the 1950s to 2008. We chose these values because they represent a "normal" situation before the tremendous shocks associated with the Great Recession of 2009/2009 and the Covid19 pandemic hit the US economy.

👉 a) Simulate the dynamics of Public Debt as a percentage of real GDP $(d_t)$ over 500 years.

"""

# ╔═╡ 0d524314-4eac-4c0a-980b-c639c91d33ab
md"""
!!! tip "Answer (a)"

	The level of public debt as a percentage of GDP converges to 114% of GDP, but it takes a tremendous amount of time to get there: more than 300 years.

"""

# ╔═╡ cf154894-d2e3-4e75-bfd1-b48c1036b47b
md"""
👉 b) Using the zoom functionalities of the plot, zoom over 20 years within the region comprehended between the first and the 100th year. What is the natural conclusion if you were confronted only with this slice of the process?  
"""

# ╔═╡ 708e9c02-b108-4f73-bac8-b27adde85585
md"""
!!! tip "Answer (b)"

	It looks like a straight line. It gives the impression that public debt as a percentage of GDP is explosive. But this is wrong and results from us looking at a too-short period in this process.

"""

# ╔═╡ 3889d8e4-93ca-420b-a158-4edb704babc5
md"""
👉 c) Now, assuming that the current level of $d$ is 1.25 (data for US economy in 2021-Q2), see what happens if this were the initial state in our exercise.
"""

# ╔═╡ 1b572575-510a-44f1-9f3e-62a7a93baca8
md"""
!!! tip "Answer (c)"

	Public debt will go down to the same level as in question (a) -- 114% of GDP --and it takes the same amount of time to reach that level: more than 300 years.

"""

# ╔═╡ 8e5ff175-44b4-4282-bee1-96b5d822debe
md"""
---
"""

# ╔═╡ 4c9e3c52-771a-4601-9dbc-0c267f432c2f
md"""
## Exercise 10. The US public debt sustainability 
"""

# ╔═╡ 34a44636-7b93-4ed7-b0ed-d4d213e3c09b
md"""

We will deal with the sustainability of public debt in great detail in Week 11. Until then, we will only briefly introduce the topic during the current week. It is a terribly important topic, and we call upon your basic intuition rather than knowledge. Read the following sentence, which came out in a book with the suggestive title "Sovereign Debt: A Guide for Economists and Practitioners":

>"_Under normal conditions for growth and interest rates, solvency imposes public debt to be at most equal to the present value of all future primary balances. Equivalently, primary deficits must at some point be fully offset by surpluses_." 

Debrun, Xavier and Ostry, Jonathan D. and Willems, Tim and Wyplosz, Charles (2019), "Public Debt Sustainability", in "Sovereign Debt: A Guide for Economists and Practitioners", Oxford University Press, available [here](https://www.imf.org/-/media/Files/News/Seminars/2018/091318SovDebt-conference/chapter-4-debt-sustainability.ashx)

"""

# ╔═╡ b8f0bbc9-376a-48ff-8e82-9d8ca90c6c13
md"""
👉 a) What do you think is the main point raised by Debrun _et al._ in the sentence above?
"""

# ╔═╡ 56d82260-e3e6-40b1-ab55-9d729992b2cd
md"""
!!! answer "Answer (a)"

	Their major point is that if a country has a positive level of public debt because it has experienced budget deficits over time, the level of public debt will be under control only if the government runs budget surpluses now or in the future.

"""

# ╔═╡ 48baa94f-a949-4217-8d94-17e7fd5ac144
md"""
👉 b)  In the following figure, we present the evolution of the federal budget of the USA from 1929 and 2020. Based on this single piece of evidence, what do you expect that has happened to this country's public debt?
"""

# ╔═╡ 64ac3629-c0ce-48af-bdce-d8e584a64848
md"""
!!! answer "Answer (b)"

	Since 1929, the USA very rarely experienced budget surpluses, and when they occurred, they were pretty small. Given that the country experienced budget deficits systematically, we would be inclined to think that the level of US public debt would grow permanently since 1929 without bound. 

"""

# ╔═╡ 0de092cb-5132-4e95-a271-d567438d974e
md"""
👉 c) As we saw in the previous exercise, the sustainability of public debt depends on two main macroeconomic variables: the rate of growth of real GDP $(g)$ and the real interest rate paid on public debt $(r_p)$. A standard result in macroeconomics says that if $g>r_p$, the public debt will tend to decline over time and increase if the opposite occurs. Looking at the following figure, which confronts the federal budget balance and the public debt, what do you conclude for the period between 1946 and 1981?
"""

# ╔═╡ 0aa72da5-fc22-4419-b9e4-424d922d6fff
md"""
!!! answer "Answer (c)"

	Between 1946 and 1981, the US public debt declined systematically, from 119.1% to 31% of GDP. In this period, the US government run more budget deficits than budget surpluses. So, this evidence points to the fact that public debt declined because $g>r_p$.

"""

# ╔═╡ 9ffceb85-199f-4958-8989-9ab029d7c62e
md"""
👉 d) In the following figure, we present evidence for the US economy concerning the difference between the real GDP growth rate and the yield of 10-year issued US public debt. The mean of this difference is, for the period considered (1962-2022), close to $+0.948$. What does this number tell us about public debt sustainability in the USA?
"""

# ╔═╡ 38d0df99-3966-43ed-a307-834cc7b0b3d0
md"""
!!! answer "Answer (d)"

	The number tells us that, on average, $g-r^d=0.948\%$. It means that unless the US government starts running huge primary budget deficits, the US public debt (as a percentage of GDP) tends to go down, not up.

"""

# ╔═╡ 8a0bca3c-c77b-4c7d-afbb-2252fec34897
md"""
---
"""

# ╔═╡ 8ae07934-4958-4bf8-9270-e5facadbb1de
md"""
## Exercise 11. The fiscal multiplier
"""

# ╔═╡ caff3b80-96b0-4e3f-a563-c1cdfe81de1c
md"""
Consider our standard macroeconomic model that we have developed since week 4: 

$$\begin{align*}
\text{AD}: \quad Y &=m \cdot \overline{A}-m \cdot \phi \cdot(\overline{r}+\lambda \pi) \\[3mm]           
\text{AS}: \quad  \pi &=\pi^{e}+\gamma\left(Y-Y^{P}\right)+\rho \\[3mm]  
\text{MP}: \quad r&=\overline{r}+\lambda \pi \\[3mm]  
\text{Fisher Eq.}: \quad i& =r +\pi
\end{align*}$$

and the following information concerning exogenous variables and parameters:

$$\begin{aligned}
&\overline{A}=7.6 \ , \ m=2.0 \ , \ \phi=0.2 \ , \ \overline{r}=2.0 \ , \ \lambda=0.5 \ , \ \pi^{e}=2.0 \ , \ \gamma= 4.5 \ , \ Y^{P}=14.0 \ , \ \rho=0.
\end{aligned}$$

Note that $\left\{\pi^{e}, \overline{r}, r, i , \rho \right\}$ are rates measured in percentage points
(e.g., $2.0$ means $2.0\%$), while $\{Y, Y^P , \overline{A}\}$ are measured in trillion dollars. In the solutions below, to avoid doubt, we put the symbol % only at the final step of the solutions.

👉 a) In the previous weeks, we saw that the multiplier of the Autonomous Aggregate demand $(\overline{A})$ -- which we named by $m$ -- is equal to $m=2$. This fact tells us how much the Aggregate Demand is multiplied if $(\overline{A})$ changes by 1 unit. It involves only the demand side of the economy. To calculate the **_fiscal multiplier_**, we must consider both the demand and supply sides. We can calculate it by imposing the usual condition AD=AS and solving for $Y$. Let us do it:

$$Y=\frac{m}{1+m \phi \lambda \gamma} \bar{A}-\ldots \ \ ; \qquad \Rightarrow \qquad m^{g}=\frac{\partial Y}{\partial \bar{A}}=\frac{m}{1+m \phi \lambda \gamma}$$

Calculate the value of the fiscal multiplier considering the parameter's values above.

"""

# ╔═╡ 06566ed6-b58c-479a-a01f-d0ca5509673c
md"""
!!! note

	We must pass the required data to the notebook to solve this exercise. We do that in the following two cells, one involving $(m, \phi)$. The other uses sliders to define $(\gamma, \lambda)$. 

"""

# ╔═╡ c5944406-8db8-41ce-b916-53ac918993b2
m = 2 ; ϕ = 0.2 ;

# ╔═╡ c3f60a71-88e7-4874-a1b0-f1b34efb3dc1
md"""
To solve question (a):
"""

# ╔═╡ 00fdadd3-4c2a-46ab-9dc1-1312323e9dcc
md"""
!!! tip "Answer (a)"

	The fiscal multiplier is: $m^g = 1.05263$

"""

# ╔═╡ 25d1c90a-399a-4fc9-b0dc-c0bd2f6a8161
md"""
!!! note

	This year, we skipped the specific case of the fiscal multiplier in the Zero Lower Bound (ZLB). Therefore, students are advised to skip the following question. 

"""

# ╔═╡ 5c9acd31-75fd-4642-8c35-12942807248b
md"""
👉 b) Where is the fiscal multiplier higher: in the normal region or the ZLB?  
"""

# ╔═╡ 2354fb88-c3fe-4d7f-948d-503a664b4bf5
md"""
!!! tip "Answer (b)"

	The fiscal multiplier achieves its higher value in the ZLB region. Notice that in the ZLB, the AS is not constraining Aggregate Demand, so it is as if $\gamma = 0$. Using the sliders above, check what happens to the fiscal multiplier by setting $\gamma = 0$. The result is the same if we put $\lambda =0$, which occurs in the ZLB because the central bank does not change its interest rate ($i=0$) no matter what happens to inflation in this zone. If any of these two choices are made, then 

	$$m^g=m=2$$ 

	which is the highest value for the fiscal multiplier.

"""

# ╔═╡ fe261d67-0f45-47ed-94a1-c82bd83561f6
md"""
---
"""

# ╔═╡ 8d2ee006-bca3-4399-890d-6d5ec88bab20
md"""
## Exercise 12. Income taxes in the USA
"""

# ╔═╡ f717da14-6d84-4b37-9fa6-b0d7734ed23e
md"""
The figure below presents the evolution of income taxation in the USA since the Second World War. It covers both corporate income taxation (its highest bracket) and individual income taxation (the highest and lowest brackets).

What is the expected impact on public revenues from the evolution displayed in the two panels below?


"""

# ╔═╡ 70d8e543-8eee-4cca-a807-3ad5b6536926
md"""
!!! tip "Answer"

	If the dramatic reduction in the top brackets of income taxation (individual and corporate) would cause higher GDP growth, then this could lead to higher public revenues derived from that growth. On the other hand, the reduction in income tax rates will lead to a decline in public revenues. The net effect could be positive or negative.

	However, if the reduction in taxation does not lead to higher GDP growth, then this will lead to a significant decline in public revenues from income taxation. 

"""

# ╔═╡ 286d5694-9dda-4204-ae5c-e4298448bc1c
md"""
---
"""

# ╔═╡ c4b6ef21-c519-4e86-aac9-7a442c3e0e73
md"""
## Exercise 13. Expansionary austerity (not covered this year)
"""

# ╔═╡ 85f13236-5a4c-48a1-8f4d-c7c3143a78af
md"""

The "Expansionary Austerity" theory tells us that strong, decisive, and credible cuts in public spending lead to higher real GDP growth rates. The US economy is an excellent example to test this theory for three reasons: (i) it has a stable social and economic structure, (ii) the evolution of public debt as a percentage of GDP is extremely interesting (it has a U shape, from the early 1930s up to now; see Exercise 9), (iii) it has now one of highest ratios of public debt to GDP ($144.15\%$ in 2022). 

By looking at the first plot in Exercise 9 and the plot presented below, can one find any significant favorable evidence for this theory, as far as the US is concerned?

"""

# ╔═╡ 96dd1861-4f90-4264-8e8b-6780dbcd1a04
md"""
!!! tip "Suggested answer"

	The "Expansionary Austerity" theory of Alberto Alesina, apparently, can not find support in the case of the US economy. There is no clear evidence that positive budget surpluses lead to higher GDP growth rates, as shown in the first plot of Exercise 9. This fact is also confirmed by the correlation coefficient between GDP growth and the public budget as a percentage of GDP is very close to zero (0.056). 

	Good macroeconomics needs intellectual honesty and good common sense. Notice that if we wanted to twist the conclusion Alesina defended in his famous theory, we could use the data for the US in a particular period: 1949-1974. In this period, the US economy experienced high GDP growth rates. Public debt as a percentage of GDP came down systematically, and let us see how federal budget deficits behaved. In the plot above, we can easily see that, on average, the higher the GDP growth rate is, the more negative the federal budget deficit as a percentage of GDP will be. This fact is supported by a correlation coefficient between those two variables that is very different from zero: it equals $-0.235$.

	However, we get this conclusion only because **_we choose a particular period_**. If we choose another specific period, we may get something completely different. If we keep playing this kind of game, confirming any theory of the type "anything goes" will not be difficult. Be aware of questionable intellectual honesty and bad common sense in macroeconomics.

"""


# ╔═╡ 6ab969af-cd4e-401f-901e-a1d556acb676
md"""
____________________________________________________________________________________________________________
"""

# ╔═╡ d8c9a36b-af76-4e44-a291-253b57e0bfea
md"""
## Exercise 14. Ricardian equivalence (opcional)
"""

# ╔═╡ 1153d9ac-535a-422d-9926-56cba019e9a0
md"""

*From the textbook*


In 2001 and 2003, Congress passed legislation put forth by the George W. Bush administration to permanently lower taxes. What does Ricardian equivalence predict
should have happened to household saving?

"""

# ╔═╡ cdf1daf0-88ac-480d-b374-ac8e472ec571
md"""
	!!! tip

	    *From the textbook.* 

		"According to Ricardian equivalence, household saving should have risen to pay for the future tax increases that the higher budget deficits would require. Instead, household saving, which was at an already-low 3.5% of personal disposable income in 2003, fell even further to an average of 2.2% from 2004 to 2007. At first glance, households did not behave as Ricardian equivalence suggests. Household behavior appears to have been in line with the traditional view of how fiscal policy and government deficits affect the economy." (page 453) 

	"""

# ╔═╡ 8a8d2465-dda7-47a3-8e39-6d341a8cc69d
md"""
---
"""

# ╔═╡ 7c2c73d7-adbe-4b15-8b6a-d6101caf7cc2
md"""
## Exercise 15. Budget deficits, inflation, and GDP growth ⛔
"""

# ╔═╡ 6140a599-0906-4fea-84d2-9df816c2179e
md"""
!!! note

	This exercise is about a highly controversial topic. **_Students are advised to skip this kind of problem_** at this level of learning macroeconomics. This type of exercise is a beautiful one for more advanced levels of macroeconomics.

"""

# ╔═╡ a20b8b13-35d4-4535-9c39-9ad03e20aba7
md"""

*From the textbook*

During the 1970s, most Latin American countries ran huge budget deficits. As their governments resorted to printing money (increasing the money supply) to pay for these deficits, very high inflation rates resulted. As a consequence, real GDP declined or remained constant during the 1980s. 

👉 a) Comment on the relationship between budget deficits, inflation, and real GDP growth, for the USA economy, during the period bwteen 1948 and 2008 (before the Great Recession and Covid19 shocks).

"""

# ╔═╡ cef1564f-0099-4ddd-823a-e597301a004a
md"""
The cross-correlation coefficient between the budget deficit and the inflation rate is:
"""

# ╔═╡ 16941fb4-9f81-44c9-929e-f76f573eb997
md"""
The cross-correlation coefficient between the budget deficit and the GDP growth rate is:
"""

# ╔═╡ be62bbd6-2610-49f6-96f7-8aa395a697d1
md"""
!!! tip "Suggested answer (a)"

	Looking at the two previous plots, we can see that the US economy does not display the same type of behavior as described for Latin America in the 1970s. We can make four points:

	**(i)** Inflation seems to have no significant relationship with the public budget for this period. Looking at the cross-correlation coefficient ($-0.096$, see value above), we could conclude that a higher budget deficit by 1 percentage point (more negative values for the budget) would lead, on average, to a tiny increase in inflation by +0.096 percentage points. This fact is quite different from what happened in Latin America in the 1970s;

	**(ii)** The rate of growth of GDP seems to have a positive (but minimal) relationship with budget surpluses (cross-correlation coefficient is +0.056). This fact means that, on average, if the budget surplus increases by 1 percentage point, then the real GDP growth rate increases by +0.056 percentage points.

	**(iii)** Therefore, for the period under consideration (a period which deliberately excludes the two big shocks to the US economy after 2008: the Great Recession and the Covid19 pandemic), the federal budget does not indicate to have a clear impact on both the growth rate of real GDP and the rate of inflation. At least, as far as the US economy is concerned, it seems that budget deficits have no apparent relationship with short-term business cycles.

	**(iv)** In the postwar period, there was no clear relationship between budget deficits and inflation or economic growth in the United States. Yet this occurred in a scenario of responsible fiscal policy (only in 1 year did the deficit exceed 5 percent of output). However, it cannot be taken as an indicator that the same would occur in a scenario of irresponsible fiscal policy (as was the case in many Latin American countries in the 1970s).

"""

# ╔═╡ f0de9aba-89a0-4026-8f2e-3b84f422b446
md"""
👉 b) Suppose that your findings do not corroborate any significant relationship between budget deficits and economic growth, nor any significant relationship regarding inflation. Does it mean that budget deficits are entirely alien to short-term business cycles?
"""

# ╔═╡ a7ce0a8c-e756-48cf-81ca-86f366a47b8a
md"""
!!! tip "Answer (b)"

	The previous conclusion may be valid for the period 1948-2008. However, it is not valid for the period afterward, which includes the two most significant external shocks since the 1930s: the Great Recession of 2008-2009 and the Covid19 pandemic. The plot below shows that the primary budget quickly responded to those two shocks: in 2008, all primary balances came down abruptly, and the same happened in 2020.

"""

# ╔═╡ b7f7cfbc-4ccc-409b-b294-2bdc17c57454
md"""
---
"""

# ╔═╡ 55f14818-096c-406f-9dbb-b392fd266a65
md"""
## Auxiliary cells (do not delete)
"""

# ╔═╡ 5c691efe-00cf-4d1d-9719-845ebc2a33fb
md"### CSS code"

# ╔═╡ 29bf07fc-e10b-42fd-b6c3-5ecee23f532b
html"""
<style>
	@media screen {
		main {
			margin: 0 auto;
			max-width: 1600px;
    		padding-left: max(100px, 10%);
    		padding-right: max(380px, 10%); 
            # 383px to accommodate TableOfContents(aside=true)
		}
	}
</style>
"""

# ╔═╡ 9fa87191-7c90-4738-a45a-acd929c8bd1b
  TableOfContents()

# ╔═╡ fac9df03-ca73-42d7-ad0c-405ae84d9228
begin
      struct TwoColumns{L, R}
    left::L
    right::R
      end
      
      function Base.show(io, mime::MIME"text/html", tc::TwoColumns)
          write(io, """<div style="display: flex;"><div style="flex: 48%;margin-right:2%;">""")
          show(io, mime, tc.left)
          write(io, """</div><div style="flex: 48%;">""")
          show(io, mime, tc.right)
          write(io, """</div></div>""")
      end
end

# ╔═╡ 27727af4-96c6-40e4-a123-49491bb0d118
TwoColumns(
md"""

`p = `$(@bind p Slider(-5.0:0.1:5.0, default=1.0, show_value=true)) 

`rₚ = `$(@bind rₚ Slider(0.0:0.05:4.0, default=1.2, show_value=true))

""",
	
md"""

`g = `$(@bind g Slider(0.0:0.05:5.0, default=3.0, show_value=true))

`d₁ = `$(@bind d₁ Slider(0.0:0.1:2, default=0.0, show_value=true))

"""
)

# ╔═╡ df7ebe49-3f83-44e9-9e75-888ba8df529d
begin
	t = 500 # number of iterations we want to simulate
	n = length(d₁) # telling the routine how many initial conditions are in the loop
	d = [d₁  zeros(n, t-1)] # preallocation to increase performance

	for i = 1 : t-1 # the "for" loop
	     d[i+1] = p/100 + ((1.0 +rₚ/100)/(1+g/100)) *d[i] 
	end
end
 
#plot(1:t, y', xlabel = "time", label = "y(t)", marker = :circle)

# ╔═╡ 1d9b30d6-ac9f-450a-98e2-d6be556406ab
begin
layout11_91 = Layout(;	height = 450,
		title_text = "The evolution of Public Debt as % of GDP", 
		title_x =0.5,		
		hovermode="x",			
        xaxis = attr(
               title = "Years", 
               #tickformat = "%Y",
               #hoverformat = "%Y-M%m",
               #tick0 = "1997/10/01",
               ),		
        #xaxis_range = [2000, 2020],
        yaxis_title = "Debt / GDP",
        #yaxis_range=[-2, 2], 
        titlefont_size = 16)
plot(d', layout11_91)
end

# ╔═╡ c046d55c-f832-4ef5-a65b-525e16e2d6a7
TwoColumns(
md"""

`γ = `$(@bind γ Slider(0.0:0.5:9.0, default=4.5, show_value=true))

""",
	
md"""

`λ = `$(@bind λ Slider(0.0:0.1:1.0, default=0.5, show_value=true))

"""
)

# ╔═╡ cd368e23-2171-4cff-92fe-a33ecb24183e
mg = m /(1 + (m*ϕ*λ*γ)) 		# mg represents the fiscal multiplier

# ╔═╡ e90a162f-af60-40cf-a11f-11846f12b0e0
md"""
### Exercises
"""

# ╔═╡ b22e478a-1d6d-4a24-a358-e1448ef73827
begin

	md"""
	
- Population.csv
- OECD_Debt.csv
- MandatorySpending_2015.csv
- DiscretionarySpending_2025.csv
- SocialSecurityUS.csv
- PrimDefic1978_2022.csv
- US_Budget_Deficits_1929_2020.csv
- US_PublicDebt.csv
- g_versus_r.csv
- Tax_Rates.csv
- Anual_Growth_Deficit_Inflation.csv

	These CSV files must be kept in the same folder as this notebook.
	"""
	
	md"> _Data files used in this notebook_"
end

# ╔═╡ 451fa7e8-1582-4e9e-a658-799df448c30c
begin

	popul = CSV.read("Population.csv", DataFrame);
	
	
	md"> _Supporting cells for Exercise 3 (Aging in the USA and Portugal)_"
end

# ╔═╡ 7990b6c1-9b1f-4614-a4d1-5c7beded7354
let
	trace2_28 = scatter(;x= popul.Years, y= popul.Portugal, mode="markers+lines",
				marker_symbol="circle", marker_size="1",line_width= 2.3,
            	marker_color = "Blue", name="Portugal", yaxis = "y2",
				fillcolor = "blue", opacity = 0.9)#, text = period1_6)

	trace2_29 = scatter(;x= popul.Years, y= popul.USA, mode="markers+lines",
				marker_symbol="circle", marker_size="1",line_width= 2.3,
            	marker_color = "DarkRed", name="USA")#, text = period1_6)

	layout2_29 = Layout(height = 450,
			legend_orientation="h", legend=attr( x=0.05, y=0.97),

			title_text = "Projected Total Population for Portugal and the USA: 2025-2060",
			title_x = 0.5,
			hovermode = "x",

			titlefont_size = 16,

			xaxis = attr(
                		title = "Annual obervations",
						showgrid = false),
                		#tickformat = "%Y",
                		#hoverformat = "%Y-Y%y",
                		#tick0 = "1939/10/01",
                		#dtick = "M240"
				        xaxis_range = [1949, 2060],


			yaxis1 = attr(
						title = "United States: Millions",
						titlefont_color=  "DarkRed",
						tickfont_color = "DarkRed",
						#overlaying = "y",
						#side = "right",
						#yaxis1_tickvals = [1.5, 2.5, 3.5, 5.5],
						showgrid= true,
						zeroline=false,
						#yaxis1_range=[1.5 , 5.5],
    					#dtick = 5.6 / 7
						  ),

			yaxis2 = attr(
    					title = "Portugal: Millions",
    					titlefont_color=  "blue",
    					tickfont_color = "blue",
    					overlaying = "y",
    					side = "right",
						showgrid = false,
						#yaxis2_range=[-25 , 5],
						#dtick = 30 / %
						  )
						)
	p2_29 = plot([trace2_28, trace2_29], layout2_29)
	add_vrect!(p2_29, "2025", "2060",  fillcolor = "red", opacity = 0.2, line_width = 0; row=1, col=1)
	p2_29


end

# ╔═╡ d8eabf86-5534-464c-83ac-29db5ecd190d
begin

		oecd = CSV.read("OECD_Debt.csv", DataFrame);

		oecd2 = unstack(oecd, :date, :countries, :values);
	
		names(oecd2);
	
	md"> _Supporting cells for Exercise 4 (The size of public debt)_"
end

# ╔═╡ 20a0a2fb-8e3d-4a28-8c79-e239470dc747
begin
	p10=plot(x=1995:1:2022, [oecd2.PRT  oecd2.GBR  oecd2.USA  oecd2.FRA  oecd2.ESP  oecd2.DEU oecd2.CAN  oecd2.BEL ])
	restyle!(p10, name=["PRT" "GBR" "USA" "FRA" "ESP" "GER" "CAN" "BEL" ])
	
	relayout!(p10, height = 450, hovermode="x", title_text = "Public Debt as a % of GDP", title_x = 0.5, xaxis_range= [1994.5 , 2022.5],
		font_family="Calibri", font_size=14,
	
		annotations=[
        	attr(x=2004.8, y=164, font_size = 16,
            	text="Financial crisis starts",
            	showarrow=false,
            	arrowhead=1
        		),
        	attr(x=2018.1, y=164, font_size = 16,
            	text="Covid19",
            	showarrow=false,
            	yshift=0
        		)
    ])
	
	add_shape!(p10, line( x0=2007, y0=30, x1=2007, y1=170,  line=attr(color="RoyalBlue", width=1.5, dash="dashdot")))
	add_shape!(p10, line( x0=2019, y0=30, x1=2019, y1=170,  line=attr(color="RoyalBlue", width=1.5, dash="dashdot")))
	p10
	#savefig(p10, "G:\\Teaching\\Master\\Slides\\01_Introduction\\fig2022\\Debt.pdf", width =1000)
end

# ╔═╡ 17ebf257-279e-435e-800c-37c88e9a2a7f
begin

	#Mand = CSV.read("MandatorySpending.csv", DataFrame);

	Discret2025 = CSV.read("DiscretionarySpending_2025.csv", DataFrame);

	Mand2025 = CSV.read("MandatorySpending_2025.csv", DataFrame);

######################################################################
	
begin
	figDiscret = Plot(1962:2022,Discret2025[:,4], mode = "markers+lines", marker_color="navy",
            	marker_size = 7, marker_symbol = "circle", line_width = 0.5 , linewitdth=0.3, 		 Layout(hovermode="x", title_text = "Discretionary public spending as a % of GDP (US: 1962-2022)", 
				title_x=0.5, xaxis_range = [1961, 2023], yaxis_range=[5, 14],

	annotations=[
		attr(x=2000, y=13, text="9/11", showarrow=false, arrowhead=1 ),
        attr(x=2008, y=13, text="Financial crisis starts", showarrow=false, arrowhead=1 ),
        attr(x=2019, y=13, text="Covid19", showarrow=false, yshift=0 )]))

	add_shape!(figDiscret, line( x0=2001, y0=0, x1=2001, y1=14,  line=attr(color="RoyalBlue", width=1.5, dash="dashdot")), row=1, col=1)
	
	add_shape!(figDiscret, line( x0=2008, y0=0, x1=2008, y1=14,  line=attr(color="RoyalBlue", width=1.5, dash="dashdot")), row=1, col=1)
	
	add_shape!(figDiscret, line( x0=2019, y0=0, x1=2019, y1=14,  line=attr(color="RoyalBlue", width=1.5, dash="dashdot")), row=1, col=1)
	
    figDiscret
end;

###################################################################################

#savefig(figDiscret, "discretionary.pdf", width=800, height=400, scale=1.5);
	
##################################################################################
	
	md"> _Supporting cells for Exercise 5 (Mandatory vs discretionary spending)_"
end

# ╔═╡ b67b541c-e977-4e34-96bd-2ed132f88b50
let
	p = make_subplots(rows=2, cols=2,  specs=[Spec(colspan=2) missing ; Spec() Spec()],
            subplot_titles=["1. Mandatory + discretionary spending"  "3. Mandatory spending" ; 
							"2. Discretionary spending" missing],
            vertical_spacing = 0.1, horizontal_spacing = 0.07 )

    add_trace!(p, scatter(x = 1962:2024, y = Discret2025[:,4]+Mand2025[:,10], mode = "markers+lines", 			 						marker_color="navy", marker_size = 7, marker_symbol = "circle", line_width = 0.5 , 										linewitdth=0.3, name = "Total"), row=1, col=1)
	
    add_trace!(p, scatter(x=1962:2024, y = Discret2025[:,4], mode = "markers+lines", marker_color="blue",
            	marker_size = 5.5, marker_symbol = "circle", line_width = 0.5 , linewitdth=0.3, name = "Disc"), 
				row=2, col=1)
	
    add_trace!(p, scatter(x=1962:2024, y = Mand2025[:,10], mode = "markers+lines", marker_color="maroon",
            	marker_size = 5.5, marker_symbol = "circle", line_width = 0.5 , linewitdth = 0.3, 
				name = "Mand" ), 
				row=2, col=2)   

	annotations=[
		attr(x=1999.8, y=30, text="9/11", showarrow=false, arrowhead=1 ),
        attr(x=2008, y=30, text="Financial crisis starts", showarrow=false, arrowhead=1 ),
        attr(x=2019, y=30, text="Covid19", showarrow=false, yshift=0 )]

	
    relayout!(p, showlegend=true, title_text = "Mandatory and Discretionary Spending as a % of GDP in the US (1962-2022)", height = "700", xaxis_range = [1960, 2025], hovermode = "x", yaxis_range=[13 , 31])	
	
	append!(p.Plot.layout.annotations, annotations)

	add_shape!(p, line( x0=2001, y0=0, x1=2001, y1=31,  line=attr(color="RoyalBlue", width=1.5, dash="dashdot")), row=1, col=1)
	
	add_shape!(p, line( x0=2008, y0=0, x1=2008, y1=31,  line=attr(color="RoyalBlue", width=1.5, dash="dashdot")), row=1, col=1)
	
	add_shape!(p, line( x0=2019, y0=0, x1=2019, y1=31,  line=attr(color="RoyalBlue", width=1.5, dash="dashdot")), row=1, col=1)
    p
end


# ╔═╡ fd103fd5-38cd-482a-bb5a-3ab76b7e57fc
begin

	ssUS = CSV.read("SocialSecurityUS.csv", DataFrame);

###################################################################################
	
let
	
	trace10_71 = scatter(;x = 1976:2100, y = ssUS[31:end,2], 
				mode="markers+lines", name  = "T5YIFR", marker_symbol="circle",
				marker_size="5", line_width= 0.3, marker_color = "Blue")
	
	shapes10_82 = rect(["2023"], ["2100"],
                0, 1; fillcolor = "red", opacity = 0.1, line_width = 0,
               xref = 1945:2100, yref = "paper")

	layout10_82 = Layout(;
		
		title_text = "Number of workers per beneficiary of the US social security system", title_x =0.5,
		
		hovermode="x",
		
		shapes = shapes10_82,
		
		annotations = [attr(x = "1990", y = 32.5, showarrow = false, text = "Historical"), 
			       	   attr(x = "2050", y = 32.5, showarrow = false, text = "Projected") ],
		
            xaxis = attr(
               title = "Years", 
              # tickformat = "%Y",
               #hoverformat = "%Y-M%m",
               #tick0 = "2006/01/01",
               #dtick = "M12",
        ),
        xaxis_range = [1976, 2103],
		yaxis_range = [2, 3.5],
        yaxis_title = "Ratio: workers/beneficiaries",
        titlefont_size = 16)

	p10_72 = Plot([trace10_71], layout10_82)
end;
	
###################################################################################
	
	md"> _Supporting cells for Exercise 7 (Data vs the popular view)_"
end

# ╔═╡ 7645f11a-4499-4aca-be48-d1a6819bdfe6
begin
	
	trace10_71 = scatter(;x = 1945:2100, y = ssUS[:,2], 
				mode="markers+lines", name  = "T5YIFR", marker_symbol="circle",
				marker_size="5", line_width= 0.3, marker_color = "Blue")
	
	shapes10_82 = rect(["2023"], ["2100"],
                0, 1; fillcolor = "red", opacity = 0.1, line_width = 0,
               xref = 1945:2100, yref = "paper")

	layout10_82 = Layout(; height = 450,		
		title_text = "Number of workers per beneficiary of the US social security system", title_x =0.5,		
		hovermode="x",		
		shapes = shapes10_82,		
		annotations = [attr(x = "1990", y = 32.5, showarrow = false, text = "Historical"), 
			       	   attr(x = "2050", y = 32.5, showarrow = false, text = "Projected") ],
		
            xaxis = attr(title = "Years", 
              
        ),
        xaxis_range = [1943, 2103],
        yaxis_title = "Ratio: workers/beneficiaries",
        titlefont_size = 16
	)

	p10_72 = Plot([trace10_71], layout10_82)
end

# ╔═╡ d1e81238-0c86-492d-8dc4-35811783f414
begin

	period11_3 = 1978:1:2022;

	PD = CSV.read("PrimDefic1978_2022.csv", DataFrame)

	PD_summary1 = describe(PD);

	PD_summary2= describe(PD[1:31,:]);


###########################################################################################
	
	md"> _Supporting cells for Exercise 8 (Business cycles and primary deficits)_"
end

# ╔═╡ 6fa199d8-eb57-454f-8d50-45966d12396c
begin
	data2 = AbstractTrace[]
	#################
	col_idx2 = [2,3,4,5,6,7,8,9,10,11,12,13,14,15,16]
	some_colnames2 = []
		for k in col_idx2
    		push!(some_colnames2, names(PD)[k])
		end
#################
		for col in some_colnames2
    		push!(data2, scatter(x = period11_3, y = PD[!,col], name = col, mode = "markers+lines",
            	marker_size = 7, marker_symbol = "circle", line_width = 0.5 ))
		end

	layout2_12 = Layout(; height = 450,
				colorway=ColorSchemes.phase.colors[1:20:end],
				title_text = "Fig 4. Primary balances in the OECD:1978-2022", 
				title_x=0.5,
				hovermode="x",		
        		xaxis_title = "Years",
        		xaxis_range = [1977, 2023],
        		yaxis_title = "PD",
        		#yaxis_range=[-2, 2],
        		titlefont_size=16)

	p2_12 = plot(data2, layout2_12)
	add_shape!(p2_12, line( x0=1977, y0=0, x1=2023, y1=0,  line=attr(color="RoyalBlue", width=1.5, dash="line")), row=1, col=1)
	p2_12
end

# ╔═╡ f76f95c2-f228-449e-b228-91520c2cad08
begin
	trace1 = bar(; x = PD_summary1[2:16,1], y = PD_summary1[2:16,2], name = "1978-2022", barmode="stack", 	 
 				legendgroup = "1978-2022", marker_color = "Darkred")
	trace2 = bar(; x = PD_summary2[2:16,1], y = PD_summary2[2:16,2], name = "1978-2008", barmode="stack", 
 				legendgroup = "1978-2008", marker_color = "Orange")
	p1 = plot([trace1, trace2] , Layout(;barmode="group", height = 450,
				title_text = "Fig 5. Primary balance: mean between 1978-2022 (Source: OECD)", 
				title_x = 0.5, hovermode = "x"))
end


# ╔═╡ 86b6c432-b0a2-48bb-9ce2-a5e4a2affca2
begin

	data1 = AbstractTrace[]
	#################
	col_idx1 = [2,3,10,6,5]
	some_colnames1 = []
		for k in col_idx1
    		push!(some_colnames1, names(PD)[k])
		end
	#################
		for col in some_colnames1
    		push!(data1, scatter(x = period11_3, y = PD[!,col], name = col, mode = "markers+lines",
            	marker_size = 7, marker_symbol = "circle", line_width = 0.5 ))
		end

	p2_11 = plot(data1, layout2_12)

	md"> _Supporting cells for Exercise 8 - 2nd part (Business cycles and primary deficits)_"
end

# ╔═╡ bd4f4f5b-1928-4cdb-a0e3-3a437d683db0
begin

	USBudget = CSV.read("US_Budget_Deficits_1929_2024.csv", DataFrame);

	USDebt = CSV.read("US_PublicDebt.csv", DataFrame);

	period2_6 = 1939:1:2019;

	period2_5 = 1929:1:2020;

	g_vs_r= CSV.read("g_versus_r.csv", delim=";", DataFrame);

	period10= QuarterlyDate(1962, 1):Quarter(1):QuarterlyDate(2022, 4);


	######################################################################################

let
	trace2_28 = bar(;x= period2_6, y= USDebt[:,2], mode="markers+lines",
				marker_symbol="circle", marker_size="5",line_width= 0.3,
            	marker_color = "Blue", name="Budget", yaxis = "y2",
				fillcolor = "blue", opacity = 0.4)#, text = period1_6)

	trace2_29 = scatter(;x= period2_6, y= USDebt[:,3], mode="markers+lines",
				marker_symbol="circle", marker_size="6",line_width= 0.3,
            	marker_color = "DarkRed", name="Debt")#, text = period1_6)

	layout2_29 = Layout(

			title_text = "Federal Budget and Federal Debt as a % of GDP : USA (1939-2019)",
			title_x = 0.5,
			hovermode = "x",

			titlefont_size = 16,

			xaxis = attr(
                		title = "Annual obervations",
						showgrid = false),
                		#tickformat = "%Y",
                		#hoverformat = "%Y-Y%y",
                		#tick0 = "1939/10/01",
                		#dtick = "M240"
				        xaxis_range = [1938, 2020],


			yaxis1 = attr(
						title = "Public Debt/GDP",
						titlefont_color=  "DarkRed",
						tickfont_color = "DarkRed",
						#overlaying = "y",
						#side = "right",
						#yaxis1_tickvals = [1.5, 2.5, 3.5, 5.5],
						showgrid= true,
						zeroline=false,
						#yaxis1_range=[1.5 , 5.5],
    					#dtick = 5.6 / 7
						  ),

			yaxis2 = attr(
    					title = "Federal Budget/GDP",
    					titlefont_color=  "blue",
    					tickfont_color = "blue",
    					overlaying = "y",
    					side = "right",
						showgrid = false,
						#yaxis2_range=[-25 , 5],
						matches="y"
						#dtick = 30 / %
						  )
						)
	p2_29 = Plot([trace2_28, trace2_29], layout2_29)

end;


	#######################################################################################

let
	y1 = USDebt[:,2]
	y2 = USDebt[:,3]

# Create the layout with two y-axes
layout = Layout(
    title="Two Time Series with In-Sync Gridlines",
    xaxis_title="Time",
    yaxis=attr(
        title="Random Series (y1)",
        zeroline=false,
        rangemode="tozero",
        titlefont=attr(color="#4bb2c5")
    ),
    yaxis2=attr(
        title="Random Series (y2)",
        zeroline=false,
        rangemode="tozero",
        showgrid=false,
        titlefont=attr(color="#953579"),
        overlaying="y",
        side="right",
        #matches="y"
    )
)

# Create the data traces
trace1 = bar(
    x=period2_6,
    y=y1,
    mode="lines+markers",
    name="y1)",
    marker=attr(color="#4bb2c5")
)

trace2 = scatter(
    x=period2_6,
    y=y2,
    mode="lines+markers",
    name="y2)",
    yaxis="y2",
    marker=attr(color="#953579")
)

# Plot the data and layout
data = [trace1, trace2]
p55=Plot(data, layout)
	#relayout!(p55, yaxis1_tickvals = [25, 50, 75, 100, 125], yaxis2_tickvals = [-30, -20, -10, 0, 10])
	#relayout!(p55, yaxis1_tickvals = [-30, -20, -10, 0, 10], yaxis2_tickvals = [25, 50, 75, 100, 125])
	p55
end;



	###################################################################################
	
	
	md"> _Supporting cells for Exercise 9 (US public debt sustainability)_"
end

# ╔═╡ 0504c209-2fab-4624-b8df-e6f01424e985
begin
	fig48 = plot(bar(x=USBudget.Years, y=USBudget.Deficit))
	restyle!(fig48, mode="bar")
	relayout!(fig48, height = 450,
		title_text = "Federal Budget as a % of GDP : USA (1929-2024)",
		title_x = 0.5,
		hovermode = "x",
        xaxis_title = "Anual obervations",
        xaxis_range = ["1928-01-01","2025-01-01"],
        #separators = ".",
        yaxis_title = "% points",
        #yaxis_range=[-2, 2],
        titlefont_size = 16,
        bargap=0.05	)
	fig48
	#savefig(fig48, "USBudgetDefict.svg", width=1000,height = 500)
end

# ╔═╡ d1ad5341-48cc-447e-97aa-2e8b9cbe4a7e
begin
	trace2_28 = bar(;x= period2_6, y= USDebt[:,2], mode="markers+lines",
				marker_symbol="circle", marker_size="5",line_width= 0.3,
            	marker_color = "Blue", name="Budget", yaxis = "y2",
				fillcolor = "blue", opacity = 0.4)#, text = period1_6)

	trace2_29 = scatter(;x= period2_6, y= USDebt[:,3], mode="markers+lines",
				marker_symbol="circle", marker_size="6",line_width= 0.3,
            	marker_color = "DarkRed", name="Debt")#, text = period1_6)

	layout2_29 = Layout(height = 450,

			title_text = "Federal Budget and Federal Debt as a % of GDP : USA (1939-2019)",
			title_x = 0.5,
			hovermode = "x",

			titlefont_size = 16,

			xaxis = attr(
                		title = "Annual obervations",
						showgrid = false),
                		#tickformat = "%Y",
                		#hoverformat = "%Y-Y%y",
                		#tick0 = "1939/10/01",
                		#dtick = "M240"
				        xaxis_range = [1938, 2020],


			yaxis1 = attr(
						title = "Public Debt/GDP",
						titlefont_color=  "DarkRed",
						tickfont_color = "DarkRed",
						#overlaying = "y",
						#side = "right",
						#yaxis1_tickvals = [1.5, 2.5, 3.5, 5.5],
						showgrid= true,
						zeroline=false,
						yaxis1_range=[1.5 , 5.5],
    					#dtick = 5.6 / 7
						  ),

			yaxis2 = attr(
    					title = "Federal Budget/GDP",
    					titlefont_color=  "blue",
    					tickfont_color = "blue",
    					overlaying = "y",
    					side = "right",
						showgrid = false,
						yaxis2_range=[-25 , 5],
						#dtick = 30 / %
						  )
						)
	p2_29 = plot([trace2_28, trace2_29], layout2_29)

end

# ╔═╡ b92e5c9f-681c-4d21-a6fc-e1bee2a26867
mean(g_vs_r.GDP_growth-g_vs_r.Real10Y_yield)

# ╔═╡ 3c2ba7d4-3c5e-4ec5-a905-2766488b6147
let
	fig10_1 = plot(bar(x=g_vs_r.dates, y=g_vs_r.GDP_growth-g_vs_r.Real10Y_yield))
	relayout!(fig10_1, Layout(hovermode="x", height = 450,title_text="GDP growth rate minus real interest rate on US debt: 1962-Q1--2022-Q4", title_x = 0.5, titlefont_size="17", tick0 = "1960"), xaxis_range = ["1961", "2023-05-01"])
	fig10_1
end

# ╔═╡ b7ccb651-491f-41d1-a36d-43484300b88c
begin

	taxes = CSV.read("Tax_Rates.csv", DataFrame);
	
	md"> _Supporting cells for Exercise 12 (Income taxation in the USA)_"
end

# ╔═╡ c1ef1449-2bb1-4669-9536-0e28e5e9bba3
begin

	BudDef = CSV.read("Anual_Growth_Deficit_Inflation.csv", DataFrame);

	period11_6 = 1948:1:2008;

	CPI6    = BudDef[1:61,2];	
	Budget6 = BudDef[1:61,3];	
	GDP_GR6 = BudDef[1:61,4]

##################################################################################

	begin
	layout2_32 = Layout(;
					title_text="Budget as a % of GDP vs GDP growth rate: USA (1948-2008)",
					title_x = 0.5,
					#hovermode = "x",
        			yaxis_title = "Rate of growth of GDP",
        			#xaxis_range = [1960, 2020],
        			xaxis_title = "Federal Budget as a % of GDP",
        			#yaxis_range=[-2, 2],
        			titlefont_size=18)


	p2_32 = Plot(Budget6 , GDP_GR6 , mode="markers+lines", text=period11_6,
            	marker_symbol="circle", marker_size="6",line_width=0.3,
            	#marker_color="LightSteelBlue",
            	marker_color="DarkRed",layout2_32)

end;


##################################################################################

begin
	layout2_53 = Layout(; 
					title_text ="Budget as a % of GDP vs inflation (CPI): USA (1948-2008)",
					title_x = 0.5,
        			yaxis_title = "Rate of inflation",
        			#xaxis_range = [1960, 2020],
        			xaxis_title = "Federal Budget as a % of GDP",
        			#yaxis_range=[-2, 2],
        			titlefont_size=18)


	p2_53 = Plot(Budget6 , CPI6 , mode="markers+lines", text=period11_6,
            	marker_symbol="circle", marker_size="6",line_width=0.3,
            	#marker_color="LightSteelBlue",
            	marker_color="Blue",layout2_53)

end;

#####################################################################################

	PD2 = PD[27:45,:];

	period11_6a = 2004:1:2022;

##################################################################################
	
	
	md"> _Supporting cells for Exercise 15 (Budget deficits, inflation, and growth)_"
end

# ╔═╡ f54534fa-f756-446f-973a-5959bcc6795b
let

	layout2_24aa = Layout(; height = 450,
        			yaxis_title = "Percentage points",
        			#xaxis_range = [1960, 2020],
        			xaxis_title = "Individual Income Taxes (highest vs lowest brackets)",
        			#yaxis_range=[-2, 2],
        			titlefont_size=18)

	layout2_25aa = Layout(; height = 450,
        			yaxis_title = "Percentage points",
        			#xaxis_range = [1960, 2020],
        			xaxis_title = "Corporate income Taxes (highest bracket)",
        			#yaxis_range=[-2, 2],
        			titlefont_size=18)


	p2_24aa = Plot(taxes.Years, [taxes.IITTRHB taxes.IITTRLB], name = ["GDP" "fff"], mode="lines", text=period11_6,
            	marker_symbol="circle", marker_size="6",line_width=2,
            	#marker_color="LightSteelBlue",
            	line_color=["maroon" "darkred"],layout2_24aa)
	restyle!(p2_24aa, name=["Highest", "Lowest"], showlegend=false)
	
	relayout!(p2_24aa, #title_text="U.S. Individual and Corporate Income Tax Rates", title_x = 0.5, 
		showlegend=false	)

    p2_25aa = Plot(taxes.Years, taxes.CorporTax, name = "Corporate", mode="lines", text=period11_6,
            	marker_symbol="circle", marker_size="6",line_width=2,
            	#marker_color="LightSteelBlue",
            	marker_color="blue",layout2_25aa)

    p2_26aa = [p2_24aa  p2_25aa]

    relayout!(p2_26aa, height = 450, title_text="U.S. Individual and Corporate Income Tax Rates", title_x = 0.5, showlegend=false	)

    p2_26aa
end

# ╔═╡ 076197a6-7882-4e10-956e-75ed6a2eb4a2
begin

	layout2_24 = Layout(;
        			yaxis_title = "Rate of GDP growth",
        			#xaxis_range = [1960, 2020],
        			xaxis_title = "Federal Budget as a % of GDP",
        			#yaxis_range=[-2, 2],
        			titlefont_size=18)

	layout2_25 = Layout(;
        			yaxis_title = "Rate of inflation",
        			#xaxis_range = [1960, 2020],
        			xaxis_title = "Federal Budget as a % of GDP",
        			#yaxis_range=[-2, 2],
        			titlefont_size=18)


	p2_24 = plot(Budget6 , GDP_GR6, name = "GDP", mode="markers+lines", text=period11_6,
            	marker_symbol="circle", marker_size="6",line_width=0.3,
            	#marker_color="LightSteelBlue",
            	marker_color="maroon",layout2_24)

    p2_25 = plot(Budget6 , CPI6, name = "CPI", mode="markers+lines", text=period11_6,
            	marker_symbol="circle", marker_size="6",line_width=0.3,
            	#marker_color="LightSteelBlue",
            	marker_color="blue",layout2_25)

    p2_26 = [p2_24  p2_25]

    relayout!(p2_26, height = 450,
		title_text="Federal Budget, GDP growth and Inflation: USA (1948-2008)",
		title_x = 0.5,)

    p2_26

end

# ╔═╡ 202aeee1-e2c1-4d53-8d8d-d0a76761a9cb
crosscor(Budget6 , CPI6, [0]; demean=true)

# ╔═╡ 79c2a5ec-4b86-4d66-8f35-48ca70452af1
crosscor(Budget6 , GDP_GR6, [0]; demean=true)

# ╔═╡ c179210c-452f-4272-bd40-122d94f5113d
begin
	data3 = AbstractTrace[]
	#################
	col_idx3 = [2,3,4,5,6,7,8,9,10,11,12,13,14,15,16]
	some_colnames3 = []
		for k in col_idx3
    		push!(some_colnames3, names(PD2)[k])
		end
#################
		for col in some_colnames3
    		push!(data3, scatter(x = period11_6a, y = PD2[!,col], name = col, mode = "markers+lines",
            	marker_size = 7, marker_symbol = "circle", line_width = 0.5 ))
		end

	layout11_12 = Layout(; height = 450,
				colorway=ColorSchemes.phase.colors[1:20:end],
				title_text = "Primary balances in the OECD:2004-2022", 
				title_x=0.5,
				hovermode="x",		
        		xaxis_title = "Years",
        		xaxis_range = [2003.5 , 2022.5],
        		yaxis_title = "PD",
        		#yaxis_range=[-2, 2],
        		titlefont_size=16)

	p11_12 = plot(data3, layout11_12)
end

# ╔═╡ 3a52b844-966d-46ac-ad77-1cb951271aaf
begin

	CPI8    = BudDef[2:27,2];	
	Budget8 = BudDef[2:27,3];	
	GDP_GR8 = BudDef[2:27,4]

	#CPI8   = BudDef[2:34,2];	
	#Budget8 = BudDef[2:34,3];	
	#GDP_GR8 = BudDef[2:34,4]

	period11_8 = 1949:1:1974;
	#period11_8 = 1949:1:1981;
	
	
	md"> _Supporting cells for Exercise 13 (Expansionary austerity)_"
end

# ╔═╡ 36756c9a-ab87-4096-bf58-3bb8fed00e11
begin
	layout2_42 = Layout(;
					height = 600, width = 600,
					title_text="Budget as a % of GDP vs GDP growth rate: USA (1949-1974)",
					title_x = 0.5,
					#hovermode = "x",
        			yaxis_title = "GDP growth rate",
        			#xaxis_range = [1960, 2020],
        			xaxis_title = "Federal Budget as a % of GDP",
        			#yaxis_range=[-2, 2],
        			titlefont_size=18)


	p2_42 = plot(Budget8 , GDP_GR8 , mode="markers", text=period11_8,
            	marker_symbol="circle", marker_size="10",line_width=0.3,
            	#marker_color="LightSteelBlue",
            	marker_color="DarkRed",layout2_42)

end

# ╔═╡ 9edde6d4-fb17-4569-84f9-b1a11ad3519a
crosscor(Budget8 , GDP_GR8, [0]; demean=true)

# ╔═╡ 00000000-0000-0000-0000-000000000001
PLUTO_PROJECT_TOML_CONTENTS = """
[deps]
CSV = "336ed68f-0bac-5ca0-87d4-7b16caf5d00b"
ColorSchemes = "35d6a980-a343-548e-a6ea-1d62b119f2f4"
DataFrames = "a93c6f00-e57d-5684-b7b6-d8193f3e46c0"
Dates = "ade2ca70-3891-5945-98fb-dc099432e06a"
HypertextLiteral = "ac1192a8-f4b3-4bfe-ba22-af5b92cd3ab2"
PeriodicalDates = "276e7ca9-e0d7-440b-97bc-a6ae82f545b1"
PlutoPlotly = "8e989ff0-3d88-8e9f-f020-2b208a939ff0"
PlutoUI = "7f904dfe-b85e-4ff6-b463-dae2292396a8"
StatsBase = "2913bbd2-ae8a-5f71-8c99-4fb6c76f3a91"

[compat]
CSV = "~0.10.15"
ColorSchemes = "~3.31.0"
DataFrames = "~1.8.1"
HypertextLiteral = "~0.9.5"
PeriodicalDates = "~2.0.0"
PlutoPlotly = "~0.6.5"
PlutoUI = "~0.7.73"
StatsBase = "~0.34.8"
"""

# ╔═╡ 00000000-0000-0000-0000-000000000002
PLUTO_MANIFEST_TOML_CONTENTS = """
# This file is machine-generated - editing it directly is not advised

julia_version = "1.11.6"
manifest_format = "2.0"
project_hash = "ed4539150b86e11bc63fdbabecf79f6578b00278"

[[deps.AbstractPlutoDingetjes]]
deps = ["Pkg"]
git-tree-sha1 = "6e1d2a35f2f90a4bc7c2ed98079b2ba09c35b83a"
uuid = "6e696c72-6542-2067-7265-42206c756150"
version = "1.3.2"

[[deps.AliasTables]]
deps = ["PtrArrays", "Random"]
git-tree-sha1 = "9876e1e164b144ca45e9e3198d0b689cadfed9ff"
uuid = "66dad0bd-aa9a-41b7-9441-69ab47430ed8"
version = "1.1.3"

[[deps.ArgTools]]
uuid = "0dad84c5-d112-42e6-8d28-ef12dabb789f"
version = "1.1.2"

[[deps.Artifacts]]
uuid = "56f22d72-fd6d-98f1-02f0-08ddc0907c33"
version = "1.11.0"

[[deps.Base64]]
uuid = "2a0f44e3-6c83-55bd-87e4-b1978d98bd5f"
version = "1.11.0"

[[deps.CSV]]
deps = ["CodecZlib", "Dates", "FilePathsBase", "InlineStrings", "Mmap", "Parsers", "PooledArrays", "PrecompileTools", "SentinelArrays", "Tables", "Unicode", "WeakRefStrings", "WorkerUtilities"]
git-tree-sha1 = "deddd8725e5e1cc49ee205a1964256043720a6c3"
uuid = "336ed68f-0bac-5ca0-87d4-7b16caf5d00b"
version = "0.10.15"

[[deps.CodecZlib]]
deps = ["TranscodingStreams", "Zlib_jll"]
git-tree-sha1 = "962834c22b66e32aa10f7611c08c8ca4e20749a9"
uuid = "944b1d66-785c-5afd-91f1-9de20f533193"
version = "0.7.8"

[[deps.ColorSchemes]]
deps = ["ColorTypes", "ColorVectorSpace", "Colors", "FixedPointNumbers", "PrecompileTools", "Random"]
git-tree-sha1 = "b0fd3f56fa442f81e0a47815c92245acfaaa4e34"
uuid = "35d6a980-a343-548e-a6ea-1d62b119f2f4"
version = "3.31.0"

[[deps.ColorTypes]]
deps = ["FixedPointNumbers", "Random"]
git-tree-sha1 = "67e11ee83a43eb71ddc950302c53bf33f0690dfe"
uuid = "3da002f7-5984-5a60-b8a6-cbb66c0b333f"
version = "0.12.1"
weakdeps = ["StyledStrings"]

    [deps.ColorTypes.extensions]
    StyledStringsExt = "StyledStrings"

[[deps.ColorVectorSpace]]
deps = ["ColorTypes", "FixedPointNumbers", "LinearAlgebra", "Requires", "Statistics", "TensorCore"]
git-tree-sha1 = "8b3b6f87ce8f65a2b4f857528fd8d70086cd72b1"
uuid = "c3611d14-8923-5661-9e6a-0046d554d3a4"
version = "0.11.0"

    [deps.ColorVectorSpace.extensions]
    SpecialFunctionsExt = "SpecialFunctions"

    [deps.ColorVectorSpace.weakdeps]
    SpecialFunctions = "276daf66-3868-5448-9aa4-cd146d93841b"

[[deps.Colors]]
deps = ["ColorTypes", "FixedPointNumbers", "Reexport"]
git-tree-sha1 = "37ea44092930b1811e666c3bc38065d7d87fcc74"
uuid = "5ae59095-9a9b-59fe-a467-6f913c188581"
version = "0.13.1"

[[deps.Compat]]
deps = ["TOML", "UUIDs"]
git-tree-sha1 = "9d8a54ce4b17aa5bdce0ea5c34bc5e7c340d16ad"
uuid = "34da2185-b29b-5c13-b0c7-acf172513d20"
version = "4.18.1"
weakdeps = ["Dates", "LinearAlgebra"]

    [deps.Compat.extensions]
    CompatLinearAlgebraExt = "LinearAlgebra"

[[deps.CompilerSupportLibraries_jll]]
deps = ["Artifacts", "Libdl"]
uuid = "e66e0078-7015-5450-92f7-15fbd957f2ae"
version = "1.1.1+0"

[[deps.Crayons]]
git-tree-sha1 = "249fe38abf76d48563e2f4556bebd215aa317e15"
uuid = "a8cc5b0e-0ffa-5ad4-8c14-923d3ee1735f"
version = "4.1.1"

[[deps.DataAPI]]
git-tree-sha1 = "abe83f3a2f1b857aac70ef8b269080af17764bbe"
uuid = "9a962f9c-6df0-11e9-0e5d-c546b8b5ee8a"
version = "1.16.0"

[[deps.DataFrames]]
deps = ["Compat", "DataAPI", "DataStructures", "Future", "InlineStrings", "InvertedIndices", "IteratorInterfaceExtensions", "LinearAlgebra", "Markdown", "Missings", "PooledArrays", "PrecompileTools", "PrettyTables", "Printf", "Random", "Reexport", "SentinelArrays", "SortingAlgorithms", "Statistics", "TableTraits", "Tables", "Unicode"]
git-tree-sha1 = "d8928e9169ff76c6281f39a659f9bca3a573f24c"
uuid = "a93c6f00-e57d-5684-b7b6-d8193f3e46c0"
version = "1.8.1"

[[deps.DataStructures]]
deps = ["OrderedCollections"]
git-tree-sha1 = "e357641bb3e0638d353c4b29ea0e40ea644066a6"
uuid = "864edb3b-99cc-5e75-8d2d-829cb0a9cfe8"
version = "0.19.3"

[[deps.DataValueInterfaces]]
git-tree-sha1 = "bfc1187b79289637fa0ef6d4436ebdfe6905cbd6"
uuid = "e2d170a0-9d28-54be-80f0-106bbe20a464"
version = "1.0.0"

[[deps.Dates]]
deps = ["Printf"]
uuid = "ade2ca70-3891-5945-98fb-dc099432e06a"
version = "1.11.0"

[[deps.DelimitedFiles]]
deps = ["Mmap"]
git-tree-sha1 = "9e2f36d3c96a820c678f2f1f1782582fcf685bae"
uuid = "8bb1440f-4735-579b-a4ab-409b98df4dab"
version = "1.9.1"

[[deps.DocStringExtensions]]
git-tree-sha1 = "7442a5dfe1ebb773c29cc2962a8980f47221d76c"
uuid = "ffbed154-4ef7-542d-bbb7-c09d3a79fcae"
version = "0.9.5"

[[deps.Downloads]]
deps = ["ArgTools", "FileWatching", "LibCURL", "NetworkOptions"]
uuid = "f43a241f-c20a-4ad4-852c-f6b1247861c6"
version = "1.6.0"

[[deps.FilePathsBase]]
deps = ["Compat", "Dates"]
git-tree-sha1 = "3bab2c5aa25e7840a4b065805c0cdfc01f3068d2"
uuid = "48062228-2e41-5def-b9a4-89aafe57970f"
version = "0.9.24"
weakdeps = ["Mmap", "Test"]

    [deps.FilePathsBase.extensions]
    FilePathsBaseMmapExt = "Mmap"
    FilePathsBaseTestExt = "Test"

[[deps.FileWatching]]
uuid = "7b1f6079-737a-58dc-b8bc-7a2ca5c1b5ee"
version = "1.11.0"

[[deps.FixedPointNumbers]]
deps = ["Statistics"]
git-tree-sha1 = "05882d6995ae5c12bb5f36dd2ed3f61c98cbb172"
uuid = "53c48c17-4a7d-5ca2-90c5-79b7896eea93"
version = "0.8.5"

[[deps.Future]]
deps = ["Random"]
uuid = "9fa8497b-333b-5362-9e8d-4d0656e87820"
version = "1.11.0"

[[deps.HashArrayMappedTries]]
git-tree-sha1 = "2eaa69a7cab70a52b9687c8bf950a5a93ec895ae"
uuid = "076d061b-32b6-4027-95e0-9a2c6f6d7e74"
version = "0.2.0"

[[deps.Hyperscript]]
deps = ["Test"]
git-tree-sha1 = "179267cfa5e712760cd43dcae385d7ea90cc25a4"
uuid = "47d2ed2b-36de-50cf-bf87-49c2cf4b8b91"
version = "0.0.5"

[[deps.HypertextLiteral]]
deps = ["Tricks"]
git-tree-sha1 = "7134810b1afce04bbc1045ca1985fbe81ce17653"
uuid = "ac1192a8-f4b3-4bfe-ba22-af5b92cd3ab2"
version = "0.9.5"

[[deps.IOCapture]]
deps = ["Logging", "Random"]
git-tree-sha1 = "0ee181ec08df7d7c911901ea38baf16f755114dc"
uuid = "b5f81e59-6552-4d32-b1f0-c071b021bf89"
version = "1.0.0"

[[deps.InlineStrings]]
git-tree-sha1 = "8f3d257792a522b4601c24a577954b0a8cd7334d"
uuid = "842dd82b-1e85-43dc-bf29-5d0ee9dffc48"
version = "1.4.5"

    [deps.InlineStrings.extensions]
    ArrowTypesExt = "ArrowTypes"
    ParsersExt = "Parsers"

    [deps.InlineStrings.weakdeps]
    ArrowTypes = "31f734f8-188a-4ce0-8406-c8a06bd891cd"
    Parsers = "69de0a69-1ddd-5017-9359-2bf0b02dc9f0"

[[deps.InteractiveUtils]]
deps = ["Markdown"]
uuid = "b77e0a4c-d291-57a0-90e8-8db25a27a240"
version = "1.11.0"

[[deps.InvertedIndices]]
git-tree-sha1 = "6da3c4316095de0f5ee2ebd875df8721e7e0bdbe"
uuid = "41ab1584-1d38-5bbf-9106-f11c6c58b48f"
version = "1.3.1"

[[deps.IrrationalConstants]]
git-tree-sha1 = "b2d91fe939cae05960e760110b328288867b5758"
uuid = "92d709cd-6900-40b7-9082-c6be49f344b6"
version = "0.2.6"

[[deps.IteratorInterfaceExtensions]]
git-tree-sha1 = "a3f24677c21f5bbe9d2a714f95dcd58337fb2856"
uuid = "82899510-4779-5014-852e-03e436cf321d"
version = "1.0.0"

[[deps.JSON]]
deps = ["Dates", "Mmap", "Parsers", "Unicode"]
git-tree-sha1 = "31e996f0a15c7b280ba9f76636b3ff9e2ae58c9a"
uuid = "682c06a0-de6a-54ab-a142-c8b1cf79cde6"
version = "0.21.4"

[[deps.LaTeXStrings]]
git-tree-sha1 = "dda21b8cbd6a6c40d9d02a73230f9d70fed6918c"
uuid = "b964fa9f-0449-5b57-a5c2-d3ea65f4040f"
version = "1.4.0"

[[deps.LibCURL]]
deps = ["LibCURL_jll", "MozillaCACerts_jll"]
uuid = "b27032c2-a3e7-50c8-80cd-2d36dbcbfd21"
version = "0.6.4"

[[deps.LibCURL_jll]]
deps = ["Artifacts", "LibSSH2_jll", "Libdl", "MbedTLS_jll", "Zlib_jll", "nghttp2_jll"]
uuid = "deac9b47-8bc7-5906-a0fe-35ac56dc84c0"
version = "8.6.0+0"

[[deps.LibGit2]]
deps = ["Base64", "LibGit2_jll", "NetworkOptions", "Printf", "SHA"]
uuid = "76f85450-5226-5b5a-8eaa-529ad045b433"
version = "1.11.0"

[[deps.LibGit2_jll]]
deps = ["Artifacts", "LibSSH2_jll", "Libdl", "MbedTLS_jll"]
uuid = "e37daf67-58a4-590a-8e99-b0245dd2ffc5"
version = "1.7.2+0"

[[deps.LibSSH2_jll]]
deps = ["Artifacts", "Libdl", "MbedTLS_jll"]
uuid = "29816b5a-b9ab-546f-933c-edad1886dfa8"
version = "1.11.0+1"

[[deps.Libdl]]
uuid = "8f399da3-3557-5675-b5ff-fb832c97cbdb"
version = "1.11.0"

[[deps.LinearAlgebra]]
deps = ["Libdl", "OpenBLAS_jll", "libblastrampoline_jll"]
uuid = "37e2e46d-f89d-539d-b4ee-838fcccc9c8e"
version = "1.11.0"

[[deps.LogExpFunctions]]
deps = ["DocStringExtensions", "IrrationalConstants", "LinearAlgebra"]
git-tree-sha1 = "13ca9e2586b89836fd20cccf56e57e2b9ae7f38f"
uuid = "2ab3a3ac-af41-5b50-aa03-7779005ae688"
version = "0.3.29"

    [deps.LogExpFunctions.extensions]
    LogExpFunctionsChainRulesCoreExt = "ChainRulesCore"
    LogExpFunctionsChangesOfVariablesExt = "ChangesOfVariables"
    LogExpFunctionsInverseFunctionsExt = "InverseFunctions"

    [deps.LogExpFunctions.weakdeps]
    ChainRulesCore = "d360d2e6-b24c-11e9-a2a3-2a2ae2dbcce4"
    ChangesOfVariables = "9e997f8a-9a97-42d5-a9f1-ce6bfc15e2c0"
    InverseFunctions = "3587e190-3f89-42d0-90ee-14403ec27112"

[[deps.Logging]]
uuid = "56ddb016-857b-54e1-b83d-db4d58db5568"
version = "1.11.0"

[[deps.MIMEs]]
git-tree-sha1 = "c64d943587f7187e751162b3b84445bbbd79f691"
uuid = "6c6e2e6c-3030-632d-7369-2d6c69616d65"
version = "1.1.0"

[[deps.Markdown]]
deps = ["Base64"]
uuid = "d6f4376e-aef5-505a-96c1-9c027394607a"
version = "1.11.0"

[[deps.MbedTLS_jll]]
deps = ["Artifacts", "Libdl"]
uuid = "c8ffd9c3-330d-5841-b78e-0817d7145fa1"
version = "2.28.6+0"

[[deps.Missings]]
deps = ["DataAPI"]
git-tree-sha1 = "ec4f7fbeab05d7747bdf98eb74d130a2a2ed298d"
uuid = "e1d29d7a-bbdc-5cf2-9ac0-f12de2c33e28"
version = "1.2.0"

[[deps.Mmap]]
uuid = "a63ad114-7e13-5084-954f-fe012c677804"
version = "1.11.0"

[[deps.MozillaCACerts_jll]]
uuid = "14a3606d-f60d-562e-9121-12d972cd8159"
version = "2023.12.12"

[[deps.NetworkOptions]]
uuid = "ca575930-c2e3-43a9-ace4-1e988b2c1908"
version = "1.2.0"

[[deps.OpenBLAS_jll]]
deps = ["Artifacts", "CompilerSupportLibraries_jll", "Libdl"]
uuid = "4536629a-c528-5b80-bd46-f80d51c5b363"
version = "0.3.27+1"

[[deps.OrderedCollections]]
git-tree-sha1 = "05868e21324cede2207c6f0f466b4bfef6d5e7ee"
uuid = "bac558e1-5e72-5ebc-8fee-abe8a469f55d"
version = "1.8.1"

[[deps.Parameters]]
deps = ["OrderedCollections", "UnPack"]
git-tree-sha1 = "34c0e9ad262e5f7fc75b10a9952ca7692cfc5fbe"
uuid = "d96e819e-fc66-5662-9728-84c9c7592b0a"
version = "0.12.3"

[[deps.Parsers]]
deps = ["Dates", "PrecompileTools", "UUIDs"]
git-tree-sha1 = "7d2f8f21da5db6a806faf7b9b292296da42b2810"
uuid = "69de0a69-1ddd-5017-9359-2bf0b02dc9f0"
version = "2.8.3"

[[deps.PeriodicalDates]]
deps = ["Dates", "Printf", "RecipesBase"]
git-tree-sha1 = "e616941f8093e50a373e7d51875143213f82eca4"
uuid = "276e7ca9-e0d7-440b-97bc-a6ae82f545b1"
version = "2.0.0"

[[deps.Pkg]]
deps = ["Artifacts", "Dates", "Downloads", "FileWatching", "LibGit2", "Libdl", "Logging", "Markdown", "Printf", "Random", "SHA", "TOML", "Tar", "UUIDs", "p7zip_jll"]
uuid = "44cfe95a-1eb2-52ea-b672-e2afdf69b78f"
version = "1.11.0"
weakdeps = ["REPL"]

    [deps.Pkg.extensions]
    REPLExt = "REPL"

[[deps.PlotlyBase]]
deps = ["ColorSchemes", "Colors", "Dates", "DelimitedFiles", "DocStringExtensions", "JSON", "LaTeXStrings", "Logging", "Parameters", "Pkg", "REPL", "Requires", "Statistics", "UUIDs"]
git-tree-sha1 = "28278bb0053da0fd73537be94afd1682cc5a0a83"
uuid = "a03496cd-edff-5a9b-9e67-9cda94a718b5"
version = "0.8.21"

    [deps.PlotlyBase.extensions]
    DataFramesExt = "DataFrames"
    DistributionsExt = "Distributions"
    IJuliaExt = "IJulia"
    JSON3Ext = "JSON3"

    [deps.PlotlyBase.weakdeps]
    DataFrames = "a93c6f00-e57d-5684-b7b6-d8193f3e46c0"
    Distributions = "31c24e10-a181-5473-b8eb-7969acd0382f"
    IJulia = "7073ff75-c697-5162-941a-fcdaad2a7d2a"
    JSON3 = "0f8b85d8-7281-11e9-16c2-39a750bddbf1"

[[deps.PlutoPlotly]]
deps = ["AbstractPlutoDingetjes", "Artifacts", "ColorSchemes", "Colors", "Dates", "Downloads", "HypertextLiteral", "InteractiveUtils", "LaTeXStrings", "Markdown", "Pkg", "PlotlyBase", "PrecompileTools", "Reexport", "ScopedValues", "Scratch", "TOML"]
git-tree-sha1 = "8acd04abc9a636ef57004f4c2e6f3f6ed4611099"
uuid = "8e989ff0-3d88-8e9f-f020-2b208a939ff0"
version = "0.6.5"

    [deps.PlutoPlotly.extensions]
    PlotlyKaleidoExt = "PlotlyKaleido"
    UnitfulExt = "Unitful"

    [deps.PlutoPlotly.weakdeps]
    PlotlyKaleido = "f2990250-8cf9-495f-b13a-cce12b45703c"
    Unitful = "1986cc42-f94f-5a68-af5c-568840ba703d"

[[deps.PlutoUI]]
deps = ["AbstractPlutoDingetjes", "Base64", "ColorTypes", "Dates", "Downloads", "FixedPointNumbers", "Hyperscript", "HypertextLiteral", "IOCapture", "InteractiveUtils", "JSON", "Logging", "MIMEs", "Markdown", "Random", "Reexport", "URIs", "UUIDs"]
git-tree-sha1 = "3faff84e6f97a7f18e0dd24373daa229fd358db5"
uuid = "7f904dfe-b85e-4ff6-b463-dae2292396a8"
version = "0.7.73"

[[deps.PooledArrays]]
deps = ["DataAPI", "Future"]
git-tree-sha1 = "36d8b4b899628fb92c2749eb488d884a926614d3"
uuid = "2dfb63ee-cc39-5dd5-95bd-886bf059d720"
version = "1.4.3"

[[deps.PrecompileTools]]
deps = ["Preferences"]
git-tree-sha1 = "5aa36f7049a63a1528fe8f7c3f2113413ffd4e1f"
uuid = "aea7be01-6a6a-4083-8856-8a6e6704d82a"
version = "1.2.1"

[[deps.Preferences]]
deps = ["TOML"]
git-tree-sha1 = "0f27480397253da18fe2c12a4ba4eb9eb208bf3d"
uuid = "21216c6a-2e73-6563-6e65-726566657250"
version = "1.5.0"

[[deps.PrettyTables]]
deps = ["Crayons", "LaTeXStrings", "Markdown", "PrecompileTools", "Printf", "REPL", "Reexport", "StringManipulation", "Tables"]
git-tree-sha1 = "6b8e2f0bae3f678811678065c09571c1619da219"
uuid = "08abe8d2-0d0c-5749-adfa-8a2ac140af0d"
version = "3.1.0"

[[deps.Printf]]
deps = ["Unicode"]
uuid = "de0858da-6303-5e67-8744-51eddeeeb8d7"
version = "1.11.0"

[[deps.PtrArrays]]
git-tree-sha1 = "1d36ef11a9aaf1e8b74dacc6a731dd1de8fd493d"
uuid = "43287f4e-b6f4-7ad1-bb20-aadabca52c3d"
version = "1.3.0"

[[deps.REPL]]
deps = ["InteractiveUtils", "Markdown", "Sockets", "StyledStrings", "Unicode"]
uuid = "3fa0cd96-eef1-5676-8a61-b3b8758bbffb"
version = "1.11.0"

[[deps.Random]]
deps = ["SHA"]
uuid = "9a3f8284-a2c9-5f02-9a11-845980a1fd5c"
version = "1.11.0"

[[deps.RecipesBase]]
deps = ["PrecompileTools"]
git-tree-sha1 = "5c3d09cc4f31f5fc6af001c250bf1278733100ff"
uuid = "3cdcf5f2-1ef4-517c-9805-6587b60abb01"
version = "1.3.4"

[[deps.Reexport]]
git-tree-sha1 = "45e428421666073eab6f2da5c9d310d99bb12f9b"
uuid = "189a3867-3050-52da-a836-e630ba90ab69"
version = "1.2.2"

[[deps.Requires]]
deps = ["UUIDs"]
git-tree-sha1 = "62389eeff14780bfe55195b7204c0d8738436d64"
uuid = "ae029012-a4dd-5104-9daa-d747884805df"
version = "1.3.1"

[[deps.SHA]]
uuid = "ea8e919c-243c-51af-8825-aaa63cd721ce"
version = "0.7.0"

[[deps.ScopedValues]]
deps = ["HashArrayMappedTries", "Logging"]
git-tree-sha1 = "c3b2323466378a2ba15bea4b2f73b081e022f473"
uuid = "7e506255-f358-4e82-b7e4-beb19740aa63"
version = "1.5.0"

[[deps.Scratch]]
deps = ["Dates"]
git-tree-sha1 = "9b81b8393e50b7d4e6d0a9f14e192294d3b7c109"
uuid = "6c6a2e73-6563-6170-7368-637461726353"
version = "1.3.0"

[[deps.SentinelArrays]]
deps = ["Dates", "Random"]
git-tree-sha1 = "712fb0231ee6f9120e005ccd56297abbc053e7e0"
uuid = "91c51154-3ec4-41a3-a24f-3f23e20d615c"
version = "1.4.8"

[[deps.Serialization]]
uuid = "9e88b42a-f829-5b0c-bbe9-9e923198166b"
version = "1.11.0"

[[deps.Sockets]]
uuid = "6462fe0b-24de-5631-8697-dd941f90decc"
version = "1.11.0"

[[deps.SortingAlgorithms]]
deps = ["DataStructures"]
git-tree-sha1 = "64d974c2e6fdf07f8155b5b2ca2ffa9069b608d9"
uuid = "a2af1166-a08f-5f64-846c-94a0d3cef48c"
version = "1.2.2"

[[deps.SparseArrays]]
deps = ["Libdl", "LinearAlgebra", "Random", "Serialization", "SuiteSparse_jll"]
uuid = "2f01184e-e22b-5df5-ae63-d93ebab69eaf"
version = "1.11.0"

[[deps.Statistics]]
deps = ["LinearAlgebra"]
git-tree-sha1 = "ae3bb1eb3bba077cd276bc5cfc337cc65c3075c0"
uuid = "10745b16-79ce-11e8-11f9-7d13ad32a3b2"
version = "1.11.1"
weakdeps = ["SparseArrays"]

    [deps.Statistics.extensions]
    SparseArraysExt = ["SparseArrays"]

[[deps.StatsAPI]]
deps = ["LinearAlgebra"]
git-tree-sha1 = "9d72a13a3f4dd3795a195ac5a44d7d6ff5f552ff"
uuid = "82ae8749-77ed-4fe6-ae5f-f523153014b0"
version = "1.7.1"

[[deps.StatsBase]]
deps = ["AliasTables", "DataAPI", "DataStructures", "LinearAlgebra", "LogExpFunctions", "Missings", "Printf", "Random", "SortingAlgorithms", "SparseArrays", "Statistics", "StatsAPI"]
git-tree-sha1 = "064b532283c97daae49e544bb9cb413c26511f8c"
uuid = "2913bbd2-ae8a-5f71-8c99-4fb6c76f3a91"
version = "0.34.8"

[[deps.StringManipulation]]
deps = ["PrecompileTools"]
git-tree-sha1 = "725421ae8e530ec29bcbdddbe91ff8053421d023"
uuid = "892a3eda-7b42-436c-8928-eab12a02cf0e"
version = "0.4.1"

[[deps.StyledStrings]]
uuid = "f489334b-da3d-4c2e-b8f0-e476e12c162b"
version = "1.11.0"

[[deps.SuiteSparse_jll]]
deps = ["Artifacts", "Libdl", "libblastrampoline_jll"]
uuid = "bea87d4a-7f5b-5778-9afe-8cc45184846c"
version = "7.7.0+0"

[[deps.TOML]]
deps = ["Dates"]
uuid = "fa267f1f-6049-4f14-aa54-33bafae1ed76"
version = "1.0.3"

[[deps.TableTraits]]
deps = ["IteratorInterfaceExtensions"]
git-tree-sha1 = "c06b2f539df1c6efa794486abfb6ed2022561a39"
uuid = "3783bdb8-4a98-5b6b-af9a-565f29a5fe9c"
version = "1.0.1"

[[deps.Tables]]
deps = ["DataAPI", "DataValueInterfaces", "IteratorInterfaceExtensions", "OrderedCollections", "TableTraits"]
git-tree-sha1 = "f2c1efbc8f3a609aadf318094f8fc5204bdaf344"
uuid = "bd369af6-aec1-5ad0-b16a-f7cc5008161c"
version = "1.12.1"

[[deps.Tar]]
deps = ["ArgTools", "SHA"]
uuid = "a4e569a6-e804-4fa4-b0f3-eef7a1d5b13e"
version = "1.10.0"

[[deps.TensorCore]]
deps = ["LinearAlgebra"]
git-tree-sha1 = "1feb45f88d133a655e001435632f019a9a1bcdb6"
uuid = "62fd8b95-f654-4bbd-a8a5-9c27f68ccd50"
version = "0.1.1"

[[deps.Test]]
deps = ["InteractiveUtils", "Logging", "Random", "Serialization"]
uuid = "8dfed614-e22c-5e08-85e1-65c5234f0b40"
version = "1.11.0"

[[deps.TranscodingStreams]]
git-tree-sha1 = "0c45878dcfdcfa8480052b6ab162cdd138781742"
uuid = "3bb67fe8-82b1-5028-8e26-92a6c54297fa"
version = "0.11.3"

[[deps.Tricks]]
git-tree-sha1 = "311349fd1c93a31f783f977a71e8b062a57d4101"
uuid = "410a4b4d-49e4-4fbc-ab6d-cb71b17b3775"
version = "0.1.13"

[[deps.URIs]]
git-tree-sha1 = "bef26fb046d031353ef97a82e3fdb6afe7f21b1a"
uuid = "5c2747f8-b7ea-4ff2-ba2e-563bfd36b1d4"
version = "1.6.1"

[[deps.UUIDs]]
deps = ["Random", "SHA"]
uuid = "cf7118a7-6976-5b1a-9a39-7adc72f591a4"
version = "1.11.0"

[[deps.UnPack]]
git-tree-sha1 = "387c1f73762231e86e0c9c5443ce3b4a0a9a0c2b"
uuid = "3a884ed6-31ef-47d7-9d2a-63182c4928ed"
version = "1.0.2"

[[deps.Unicode]]
uuid = "4ec0a83e-493e-50e2-b9ac-8f72acf5a8f5"
version = "1.11.0"

[[deps.WeakRefStrings]]
deps = ["DataAPI", "InlineStrings", "Parsers"]
git-tree-sha1 = "b1be2855ed9ed8eac54e5caff2afcdb442d52c23"
uuid = "ea10d353-3f73-51f8-a26c-33c1cb351aa5"
version = "1.4.2"

[[deps.WorkerUtilities]]
git-tree-sha1 = "cd1659ba0d57b71a464a29e64dbc67cfe83d54e7"
uuid = "76eceee3-57b5-4d4a-8e66-0e911cebbf60"
version = "1.6.1"

[[deps.Zlib_jll]]
deps = ["Libdl"]
uuid = "83775a58-1f1d-513f-b197-d71354ab007a"
version = "1.2.13+1"

[[deps.libblastrampoline_jll]]
deps = ["Artifacts", "Libdl"]
uuid = "8e850b90-86db-534c-a0d3-1478176c7d93"
version = "5.11.0+0"

[[deps.nghttp2_jll]]
deps = ["Artifacts", "Libdl"]
uuid = "8e850ede-7688-5339-a07c-302acd2aaf8d"
version = "1.59.0+0"

[[deps.p7zip_jll]]
deps = ["Artifacts", "Libdl"]
uuid = "3f19e933-33d8-53b3-aaab-bd5110c3b7a0"
version = "17.4.0+2"
"""

# ╔═╡ Cell order:
# ╟─1c9ca8cc-71d4-452e-a181-2574e8a1ac4d
# ╟─943edb36-acc8-4b2a-94ab-c544f9eb279b
# ╟─1245aef9-18ce-42f5-af82-c2e6cce70798
# ╟─a66961e9-d653-414d-b81f-ed924c07007b
# ╟─a8b69b08-eaee-4b1d-baed-394069f62188
# ╟─4618a663-f77e-4ccc-9b00-e84dbd5beb34
# ╠═a145e2fe-a903-11eb-160f-df7ea83fa3e6
# ╟─4bc7654b-1019-48bc-8524-4ee77e1ed577
# ╟─dcd13560-0f73-4aaa-b59a-b7961d8cdc63
# ╟─df9417d2-ebe1-498c-b299-f2b8fdee1084
# ╟─d3349840-4c97-443c-9a50-f62ba4da3f17
# ╠═ff6b8641-0e63-4f3e-9400-7979a1536e80
# ╟─42e4c961-1644-495d-96a0-38a12182a1f7
# ╠═7157f8e7-9771-4b13-846b-9a3f496e2455
# ╟─26d25950-b6bf-44c8-9ee6-21102094d4c8
# ╠═ea0ae94d-bd6a-4bf5-98b2-0a9dd28b1621
# ╟─bb024642-ce35-4464-83f8-af4493d8619b
# ╟─5686fa97-4db8-4e41-9bf2-1ca265d88c91
# ╟─d4af7521-c311-497a-ada1-3b0a74b5e411
# ╟─5cef5110-edd0-4c7f-94a3-0d4b3d183c14
# ╟─912da456-9d55-4986-8e6f-56f0d1df8d63
# ╟─0311c862-313e-4ab5-b94f-9119b34ef532
# ╟─2e73dd9a-d9a5-421a-8360-475a23259695
# ╟─08a12673-d5b4-4212-ad2d-9e32305bf335
# ╟─53f53bb1-9373-4efa-827b-8a5ea111697b
# ╟─d067df9c-476d-49c1-87f9-ed3037b32a7b
# ╟─b0b4285e-9051-4c20-a8f8-5524dc9ea3f4
# ╟─a4c3f332-2fb9-4230-8634-bfdec7a27aab
# ╟─4cac2f5c-943d-423f-9f8e-e1a741ed1762
# ╟─a9885086-22ac-4ed7-99da-40cfb107a8c8
# ╟─7fca52e8-3153-4e58-8436-0ea68d010587
# ╟─7a38bfd9-7d09-491c-9193-14655e8b2013
# ╟─52b48932-96f4-4da0-ba3c-b1abeee65285
# ╟─8759d076-cc13-4d22-b0e1-98242ae64e16
# ╟─d58319c7-0156-4804-859c-99b2f8ba4d3b
# ╟─7990b6c1-9b1f-4614-a4d1-5c7beded7354
# ╟─0848497e-60ea-4953-8dc3-5815d49df237
# ╟─24105ebf-251d-4b6c-bcff-1f035559c2cf
# ╟─75df3711-3b97-4d9a-9f61-38ef759a0983
# ╟─0b024eaa-f6d1-4d5e-b110-f74f29237ed0
# ╟─20a0a2fb-8e3d-4a28-8c79-e239470dc747
# ╟─bd2a646a-7719-4fec-872a-85ef69039040
# ╟─3b1585a8-5daa-42ea-a26e-eb46993a0d53
# ╟─2da29462-367e-4d38-a2b9-01ca32e2bb73
# ╟─046ae576-7815-4bd0-970b-c993a1abe665
# ╟─40a2d136-3ccc-4e56-9408-50ea8836f501
# ╟─2f6946dc-4346-4c9c-bbe1-fe1761eb0c1b
# ╟─2371b919-ab42-4dc2-b72e-7c5c8b3c648f
# ╟─08b17da3-50a2-41ca-b75c-209b1c7489d8
# ╟─270882ce-0740-4605-b92d-502edaf68f99
# ╟─dd58f4ec-59be-4488-8392-e5a9a7a81b03
# ╟─45c669d6-7e9d-4cdc-8f8a-8e4e9b4d6382
# ╟─b67b541c-e977-4e34-96bd-2ed132f88b50
# ╟─9b452fdc-e5c0-4aaa-827d-9f4b8f59c794
# ╟─23de519c-06a4-4879-94a6-75816e191c0c
# ╟─3ada049d-a39e-4ef5-974c-3f114852cda3
# ╟─35e486b8-3ef0-4c6f-b272-566319b63f10
# ╟─d43754b1-e498-4e83-a96e-40b36f39d802
# ╟─2c7add2a-0a48-4f03-85d7-d6af603405df
# ╟─61b2eb9d-9050-4fcb-8a62-9fa9847cd311
# ╟─18103d5d-b4d2-4c27-9a0c-909bc5bb3b08
# ╟─359bd7ea-3494-41a1-b2d8-8c6a25d7a30e
# ╟─bac4b203-a7bc-47f9-93d3-daedbac11ef6
# ╟─7645f11a-4499-4aca-be48-d1a6819bdfe6
# ╟─eab8e772-694b-490b-a791-a4f77fa4e53d
# ╟─49d3bea9-96d3-4bc7-8fc9-e413f615055f
# ╟─f805a155-e72c-458d-b7fc-dcc17a8694f3
# ╟─6faef3d2-5e28-4c9b-8238-d8882eabe6cf
# ╟─6fa199d8-eb57-454f-8d50-45966d12396c
# ╟─f535efe2-5f46-4ab2-a3b8-a6968f558bbb
# ╟─46580728-1317-477d-baa8-8672aee50694
# ╟─f76f95c2-f228-449e-b228-91520c2cad08
# ╟─06564a25-6d37-4861-9e86-0e37b8a36606
# ╟─023aa779-cc6d-48d3-8ced-a1f24e0aef35
# ╟─015f2c16-2320-426f-83bb-841c1d29213c
# ╟─86cc6fca-df27-464a-92ae-65edca20f1ff
# ╟─b8ef96d5-ec35-4d0c-aef7-bef214858661
# ╟─bbd4a839-223a-4d74-a7ea-309a6036e65d
# ╟─162d4e53-51a7-45be-a238-e84094960d4c
# ╟─45a1b8d3-cf21-4b2c-abc6-b8530ce44947
# ╟─27727af4-96c6-40e4-a123-49491bb0d118
# ╟─df7ebe49-3f83-44e9-9e75-888ba8df529d
# ╟─1d9b30d6-ac9f-450a-98e2-d6be556406ab
# ╟─0d524314-4eac-4c0a-980b-c639c91d33ab
# ╟─cf154894-d2e3-4e75-bfd1-b48c1036b47b
# ╟─708e9c02-b108-4f73-bac8-b27adde85585
# ╟─3889d8e4-93ca-420b-a158-4edb704babc5
# ╟─1b572575-510a-44f1-9f3e-62a7a93baca8
# ╟─8e5ff175-44b4-4282-bee1-96b5d822debe
# ╟─4c9e3c52-771a-4601-9dbc-0c267f432c2f
# ╟─34a44636-7b93-4ed7-b0ed-d4d213e3c09b
# ╟─b8f0bbc9-376a-48ff-8e82-9d8ca90c6c13
# ╟─56d82260-e3e6-40b1-ab55-9d729992b2cd
# ╟─48baa94f-a949-4217-8d94-17e7fd5ac144
# ╟─0504c209-2fab-4624-b8df-e6f01424e985
# ╟─64ac3629-c0ce-48af-bdce-d8e584a64848
# ╟─0de092cb-5132-4e95-a271-d567438d974e
# ╟─d1ad5341-48cc-447e-97aa-2e8b9cbe4a7e
# ╟─0aa72da5-fc22-4419-b9e4-424d922d6fff
# ╟─9ffceb85-199f-4958-8989-9ab029d7c62e
# ╟─b92e5c9f-681c-4d21-a6fc-e1bee2a26867
# ╟─3c2ba7d4-3c5e-4ec5-a905-2766488b6147
# ╟─38d0df99-3966-43ed-a307-834cc7b0b3d0
# ╟─8a0bca3c-c77b-4c7d-afbb-2252fec34897
# ╟─8ae07934-4958-4bf8-9270-e5facadbb1de
# ╟─caff3b80-96b0-4e3f-a563-c1cdfe81de1c
# ╟─06566ed6-b58c-479a-a01f-d0ca5509673c
# ╠═c5944406-8db8-41ce-b916-53ac918993b2
# ╟─c046d55c-f832-4ef5-a65b-525e16e2d6a7
# ╟─c3f60a71-88e7-4874-a1b0-f1b34efb3dc1
# ╠═cd368e23-2171-4cff-92fe-a33ecb24183e
# ╟─00fdadd3-4c2a-46ab-9dc1-1312323e9dcc
# ╟─25d1c90a-399a-4fc9-b0dc-c0bd2f6a8161
# ╟─5c9acd31-75fd-4642-8c35-12942807248b
# ╟─2354fb88-c3fe-4d7f-948d-503a664b4bf5
# ╟─fe261d67-0f45-47ed-94a1-c82bd83561f6
# ╟─8d2ee006-bca3-4399-890d-6d5ec88bab20
# ╟─f717da14-6d84-4b37-9fa6-b0d7734ed23e
# ╟─f54534fa-f756-446f-973a-5959bcc6795b
# ╟─70d8e543-8eee-4cca-a807-3ad5b6536926
# ╟─286d5694-9dda-4204-ae5c-e4298448bc1c
# ╟─c4b6ef21-c519-4e86-aac9-7a442c3e0e73
# ╟─85f13236-5a4c-48a1-8f4d-c7c3143a78af
# ╟─36756c9a-ab87-4096-bf58-3bb8fed00e11
# ╠═9edde6d4-fb17-4569-84f9-b1a11ad3519a
# ╟─96dd1861-4f90-4264-8e8b-6780dbcd1a04
# ╟─6ab969af-cd4e-401f-901e-a1d556acb676
# ╟─d8c9a36b-af76-4e44-a291-253b57e0bfea
# ╟─1153d9ac-535a-422d-9926-56cba019e9a0
# ╟─cdf1daf0-88ac-480d-b374-ac8e472ec571
# ╟─8a8d2465-dda7-47a3-8e39-6d341a8cc69d
# ╟─7c2c73d7-adbe-4b15-8b6a-d6101caf7cc2
# ╟─6140a599-0906-4fea-84d2-9df816c2179e
# ╟─a20b8b13-35d4-4535-9c39-9ad03e20aba7
# ╟─076197a6-7882-4e10-956e-75ed6a2eb4a2
# ╟─cef1564f-0099-4ddd-823a-e597301a004a
# ╠═202aeee1-e2c1-4d53-8d8d-d0a76761a9cb
# ╟─16941fb4-9f81-44c9-929e-f76f573eb997
# ╠═79c2a5ec-4b86-4d66-8f35-48ca70452af1
# ╟─be62bbd6-2610-49f6-96f7-8aa395a697d1
# ╟─f0de9aba-89a0-4026-8f2e-3b84f422b446
# ╟─a7ce0a8c-e756-48cf-81ca-86f366a47b8a
# ╟─c179210c-452f-4272-bd40-122d94f5113d
# ╟─b7f7cfbc-4ccc-409b-b294-2bdc17c57454
# ╟─55f14818-096c-406f-9dbb-b392fd266a65
# ╟─e90a162f-af60-40cf-a11f-11846f12b0e0
# ╟─b22e478a-1d6d-4a24-a358-e1448ef73827
# ╟─451fa7e8-1582-4e9e-a658-799df448c30c
# ╟─d8eabf86-5534-464c-83ac-29db5ecd190d
# ╟─17ebf257-279e-435e-800c-37c88e9a2a7f
# ╟─fd103fd5-38cd-482a-bb5a-3ab76b7e57fc
# ╟─d1e81238-0c86-492d-8dc4-35811783f414
# ╟─86b6c432-b0a2-48bb-9ce2-a5e4a2affca2
# ╟─bd4f4f5b-1928-4cdb-a0e3-3a437d683db0
# ╟─b7ccb651-491f-41d1-a36d-43484300b88c
# ╟─3a52b844-966d-46ac-ad77-1cb951271aaf
# ╟─c1ef1449-2bb1-4669-9536-0e28e5e9bba3
# ╟─5c691efe-00cf-4d1d-9719-845ebc2a33fb
# ╟─29bf07fc-e10b-42fd-b6c3-5ecee23f532b
# ╟─9fa87191-7c90-4738-a45a-acd929c8bd1b
# ╟─fac9df03-ca73-42d7-ad0c-405ae84d9228
# ╟─00000000-0000-0000-0000-000000000001
# ╟─00000000-0000-0000-0000-000000000002
